const db = require('../models/db');
const {
    BadRequestError, NotFoundError, ForbiddenError, ConflictError
} = require('../config/errors');

// ===== utils =====
function normalizeResult(result) {
    if (Array.isArray(result) && Array.isArray(result[0]) && result.length >= 1) return result[0];
    if (Array.isArray(result)) return result;
    return [];
}
async function safeQuery(sql, params, conn = null) {
    const r = conn ? await conn.query(sql, params) : await db.query(sql, params);
    return normalizeResult(r);
}
function getFirstOrThrow(items, msg = 'Registro não encontrado') { if (!items || !items.length) throw new NotFoundError(msg); return items[0]; }
function numOrThrow(v, label = 'ID') { const n = Number(v); if (Number.isNaN(n)) throw new BadRequestError(`${label} inválido`); return n; }

async function isBlockedBetween(a, b) {
    const rows = await safeQuery(
        `SELECT 1 FROM blacklists WHERE (blocker_id=? AND blocked_id=?) OR (blocker_id=? AND blocked_id=?) LIMIT 1`,
        [Number(a), Number(b), Number(b), Number(a)]
    );
    return rows.length > 0;
}


module.exports = {
    // === CREATE REQUEST ===
    // C:\src\Clientes\Datetime\services\friendshipService.js
    async sendFriendRequest(senderId, receiverId) {
        senderId = numOrThrow(senderId, 'Remetente');
        receiverId = numOrThrow(receiverId, 'Destinatário');

        if (senderId === receiverId) throw new BadRequestError('Você não pode enviar solicitação para você mesmo.');
        if (await isBlockedBetween(senderId, receiverId)) throw new ForbiddenError('Não é possível interagir com este usuário.');

        const conn = await db.getConnection();
        try {
            await conn.beginTransaction();

            // 0) Já são amigos?
            const fr = await safeQuery(
                `SELECT id FROM friendships 
       WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?) 
       LIMIT 1 FOR UPDATE`,
                [senderId, receiverId, receiverId, senderId], conn
            );
            if (fr.length) throw new ConflictError('Vocês já são amigos.');

            // 1) Existe pendente NO SENTIDO INVERSO? (receiver -> sender)
            const reversePend = await safeQuery(
                `SELECT id, sender_id, receiver_id 
         FROM friendship_requests
        WHERE sender_id=? AND receiver_id=? AND status='pending'
        LIMIT 1 FOR UPDATE`,
                [receiverId, senderId], conn
            );

            if (reversePend.length) {
                // ✅ Solicitações cruzadas => aceitar automaticamente e criar amizade
                await safeQuery(
                    `UPDATE friendship_requests 
            SET status='accepted', responded_at=NOW()
          WHERE id=?`,
                    [reversePend[0].id], conn
                );

                const already = await safeQuery(
                    `SELECT id FROM friendships 
          WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?) 
          LIMIT 1 FOR UPDATE`,
                    [senderId, receiverId, receiverId, senderId], conn
                );
                if (!already.length) {
                    const [a, b] = senderId < receiverId ? [senderId, receiverId] : [receiverId, senderId];
                    await safeQuery(
                        `INSERT INTO friendships (user1_id, user2_id) VALUES (?,?)`,
                        [a, b], conn
                    );
                }

                await conn.commit();
                return {
                    message: 'Solicitações mútuas detectadas — amizade criada.',
                    becameFriends: true
                };
            }

            // 2) Existe pendente no MESMO sentido? (sender -> receiver)
            const samePend = await safeQuery(
                `SELECT id FROM friendship_requests
        WHERE sender_id=? AND receiver_id=? AND status='pending'
        LIMIT 1 FOR UPDATE`,
                [senderId, receiverId], conn
            );
            if (samePend.length) {
                // Idempotente: já existe a pendência enviada por você
                throw new ConflictError('Já existe uma solicitação pendente.');
                // (Se preferir 200 idempotente, troque por:
                // await conn.commit(); return { message:'Solicitação já enviada', pending:true };
            }

            // 3) Se havia rejeitada/cancelada, reabrir; senão criar nova
            const prev = await safeQuery(
                `SELECT id FROM friendship_requests
        WHERE ((sender_id=? AND receiver_id=?) OR (sender_id=? AND receiver_id=?))
          AND status IN ('rejected','cancelled')
        ORDER BY created_at DESC
        LIMIT 1 FOR UPDATE`,
                [senderId, receiverId, receiverId, senderId], conn
            );

            if (prev.length) {
                await safeQuery(
                    `UPDATE friendship_requests 
            SET status='pending', created_at=NOW(), responded_at=NULL
          WHERE id=?`,
                    [prev[0].id], conn
                );
            } else {
                await safeQuery(
                    `INSERT INTO friendship_requests (sender_id, receiver_id, status) 
         VALUES (?,?, 'pending')`,
                    [senderId, receiverId], conn
                );
            }

            await conn.commit();
            return { message: 'Solicitação enviada com sucesso', pending: true };
        } catch (e) {
            await conn.rollback();
            throw e;
        } finally {
            conn.release();
        }
    },

    // === RESPOND REQUEST (accept | reject | cancel) ===
    async respondToRequest(userId, requestId, action) {
        userId = numOrThrow(userId, 'Usuário');
        requestId = numOrThrow(requestId, 'Solicitação');
        const allowed = ['accept', 'reject', 'cancel'];
        if (!allowed.includes(action)) throw new BadRequestError('Ação inválida');

        const conn = await db.getConnection();
        try {
            await conn.beginTransaction();

            const rows = await safeQuery(`SELECT * FROM friendship_requests WHERE id=? FOR UPDATE`, [requestId], conn);
            const req = getFirstOrThrow(rows, 'Solicitação não encontrada');

            // bloqueio
            if (await isBlockedBetween(req.sender_id, req.receiver_id))
                throw new ForbiddenError('Não é possível interagir com este usuário');

            // Permissões
            if (action === 'cancel') {
                if (req.sender_id !== userId) throw new ForbiddenError('Somente o remetente pode cancelar.');
                if (req.status !== 'pending') throw new ConflictError('Solicitação já processada.');
                await safeQuery(`UPDATE friendship_requests SET status='cancelled' WHERE id=?`, [requestId], conn);
            } else if (action === 'reject') {
                if (req.receiver_id !== userId) throw new ForbiddenError('Somente o destinatário pode rejeitar.');
                if (req.status !== 'pending') throw new ConflictError('Solicitação já processada.');
                await safeQuery(`UPDATE friendship_requests SET status='rejected' WHERE id=?`, [requestId], conn);
            } else if (action === 'accept') {
                if (req.receiver_id !== userId) throw new ForbiddenError('Somente o destinatário pode aceitar.');
                if (req.status !== 'pending') throw new ConflictError('Solicitação já processada.');

                await safeQuery(`UPDATE friendship_requests SET status='accepted' WHERE id=?`, [requestId], conn);

                // cria amizade se ainda não existir
                const exists = await safeQuery(
                    `SELECT id FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?)`,
                    [req.sender_id, req.receiver_id, req.receiver_id, req.sender_id], conn
                );
                if (!exists.length) {
                    const [a, b] = req.sender_id < req.receiver_id ? [req.sender_id, req.receiver_id] : [req.receiver_id, req.sender_id];
                    await safeQuery(`INSERT INTO friendships (user1_id,user2_id) VALUES (?,?)`, [a, b], conn);
                }
            }

            await conn.commit();
            return { message: 'Solicitação atualizada com sucesso' };
        } catch (e) {
            await conn.rollback(); throw e;
        } finally { conn.release(); }
    },

    // === PENDING LIST / COUNT ===
    async listRequests(userId) {
        userId = numOrThrow(userId);
        return await safeQuery(
            `SELECT fr.id, u.id AS sender_id, u.username, u.profile_picture, fr.created_at, fr.status
             FROM friendship_requests fr
             JOIN users u ON fr.sender_id = u.id
             WHERE fr.receiver_id = ? AND fr.status='pending'
             ORDER BY fr.created_at DESC`,
            [userId]
        );
    },
    async countRequests(userId) {
        userId = numOrThrow(userId);
        const r = await safeQuery(`SELECT COUNT(*) AS count FROM friendship_requests WHERE receiver_id=? AND status='pending'`, [userId]);
        return r[0]?.count || 0;
    },

    // === FRIENDS LIST (com unread_count) ===
    async listFriends(userId, search = '') {
        userId = numOrThrow(userId);
        const params = [userId, userId, userId, userId, userId];
        let filter = '';
        if (search) { filter = ` AND u.username LIKE ?`; params.push(`%${search}%`); }

        const rows = await safeQuery(
            `SELECT 
                u.id, 
                u.username, 
                u.profile_picture, 
                f.created_at AS friendship_date,
                (
                  SELECT COUNT(*)
                  FROM friend_messages m
                  WHERE m.sender_id = u.id
                    AND m.receiver_id = ?
                    AND m.is_read = 0
                ) AS unread_count
             FROM friendships f
             JOIN users u 
               ON ( (f.user1_id=? AND f.user2_id=u.id) OR (f.user2_id=? AND f.user1_id=u.id) )
             WHERE NOT EXISTS (
               SELECT 1 FROM blacklists b
               WHERE (b.blocker_id = ? AND b.blocked_id = u.id) 
                  OR (b.blocker_id = u.id AND b.blocked_id = ?)
             )${filter}
             ORDER BY u.username`,
            params
        );
        return rows;
    },

    async searchFriends(userId, term) {
        userId = numOrThrow(userId);
        if (!term || term.length < 3) throw new BadRequestError('Termo de busca deve ter ao menos 3 caracteres.');
        return await safeQuery(
            `SELECT u.id, u.username, u.profile_picture,
                CASE
                  WHEN f.id IS NOT NULL THEN 'friend'
                  WHEN frs.id IS NOT NULL AND frs.status='pending' THEN 'request_sent'
                  WHEN frr.id IS NOT NULL AND frr.status='pending' THEN 'request_received'
                  ELSE 'none'
                END AS status
             FROM users u
             LEFT JOIN friendships f ON ( (f.user1_id=? AND f.user2_id=u.id) OR (f.user2_id=? AND f.user1_id=u.id) )
             LEFT JOIN friendship_requests frs ON (frs.sender_id=? AND frs.receiver_id=u.id)
             LEFT JOIN friendship_requests frr ON (frr.sender_id=u.id AND frr.receiver_id=?)
             WHERE u.id<>? AND u.username LIKE ?
             AND NOT EXISTS (
               SELECT 1 FROM blacklists b WHERE (b.blocker_id=? AND b.blocked_id=u.id) OR (b.blocker_id=u.id AND b.blocked_id=?)
             )
             ORDER BY u.username
             LIMIT 20`,
            [userId, userId, userId, userId, userId, `%${term}%`, userId, userId]
        );
    },

    // === REMOVE FRIEND ===
    async removeFriend(userId, friendId) {
        userId = numOrThrow(userId); friendId = numOrThrow(friendId);
        if (await isBlockedBetween(userId, friendId)) throw new ForbiddenError('Não é possível interagir com este usuário.');

        const conn = await db.getConnection();
        try {
            await conn.beginTransaction();

            const del = await conn.query(
                `DELETE FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?)`,
                [userId, friendId, friendId, userId]
            );
            const affected = Array.isArray(del) && del[0]?.affectedRows != null ? del[0].affectedRows : del.affectedRows;
            if (!affected) throw new NotFoundError('Amizade não encontrada.');

            await safeQuery(
                `UPDATE friendship_requests SET status='rejected'
                 WHERE ((sender_id=? AND receiver_id=?) OR (sender_id=? AND receiver_id=?))
                 AND status IN ('pending','accepted')`,
                [userId, friendId, friendId, userId], conn
            );

            await conn.commit();
            return { message: 'Amizade removida com sucesso.' };
        } catch (e) {
            await conn.rollback(); throw e;
        } finally { conn.release(); }
    },

    // === GALLERY (corrigido: amigos sempre podem ver) ===
    async getGaleria(viewedUserId, viewerId = null) {
        const uid = numOrThrow(viewedUserId, 'Usuário da galeria');

        if (viewerId != null) {
            viewerId = numOrThrow(viewerId, 'Viewer');

            // bloqueio bidirecional?
            if (await isBlockedBetween(uid, viewerId)) {
                throw new ForbiddenError('Acesso negado');
            }

            // Se é o próprio dono, libera sempre.
            if (uid === viewerId) {
                const images = await safeQuery(
                    `SELECT id, image_path, created_at FROM gallery_images WHERE id_user=? ORDER BY created_at DESC`,
                    [uid]
                );
                return { images };
            }

            // Se já são amigos, libera sempre (independe do Explorer)
            const friends = await safeQuery(
                `SELECT 1 FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?) LIMIT 1`,
                [uid, viewerId, viewerId, uid]
            );
            if (friends.length) {
                const images = await safeQuery(
                    `SELECT id, image_path, created_at FROM gallery_images WHERE id_user=? ORDER BY created_at DESC`,
                    [uid]
                );
                return { images };
            }

            // Não são amigos: aplicar regras de Explorer
            const rows = await safeQuery(`SELECT explorer, explorer_type FROM users WHERE id=?`, [uid]);
            const u = getFirstOrThrow(rows, 'Usuário não encontrado');

            if (!u.explorer) {
                throw new ForbiddenError('Este usuário não tem o Explorer ativo.');
            }
            if (u.explorer_type === 'private') {
                throw new ForbiddenError('Apenas amigos podem ver a galeria deste usuário.');
            }
            // explorer_type público -> libera
        }

        // viewerId nulo (fallback) ou público liberado
        const images = await safeQuery(
            `SELECT id, image_path, created_at FROM gallery_images WHERE id_user=? ORDER BY created_at DESC`,
            [uid]
        );
        return { images };
    },

    // === STATUS MAP ===
    async checkStatus(userId, targetUserId) {
        userId = numOrThrow(userId); targetUserId = numOrThrow(targetUserId);

        // bloqueio?
        if (await isBlockedBetween(userId, targetUserId)) {
            return { blocked: true, friendshipStatus: 'blocked', currentUserId: userId };
        }

        // amigos?
        const fr = await safeQuery(
            `SELECT id FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?)`,
            [userId, targetUserId, targetUserId, userId]
        );
        if (fr.length) return { blocked: false, friendshipStatus: 'friends', currentUserId: userId };

        // última solicitação
        const req = await safeQuery(
            `SELECT id, sender_id, receiver_id, status, created_at
             FROM friendship_requests
             WHERE ((sender_id=? AND receiver_id=?) OR (sender_id=? AND receiver_id=?))
             ORDER BY created_at DESC LIMIT 1`,
            [userId, targetUserId, targetUserId, userId]
        );

        if (req.length) {
            const r = req[0];
            if (r.status === 'pending') {
                const youAreSender = r.sender_id === userId;
                return {
                    blocked: false,
                    friendshipStatus: youAreSender ? 'pending_sent' : 'pending_received',
                    requestId: r.id,
                    currentUserId: userId
                };
            }
            if (r.status === 'accepted') {
                return { blocked: false, friendshipStatus: 'friends', currentUserId: userId };
            }
            if (r.status === 'rejected') return { blocked: false, friendshipStatus: 'rejected', requestId: r.id, currentUserId: userId };
            if (r.status === 'cancelled') return { blocked: false, friendshipStatus: 'cancelled', requestId: r.id, currentUserId: userId };
        }

        return { blocked: false, friendshipStatus: 'none', currentUserId: userId };
    },

    // === PROFILE VIEW ===
    async viewProfile(viewerId, profileId, mustBeFriends = false) {
        viewerId = numOrThrow(viewerId); profileId = numOrThrow(profileId);
        if (await isBlockedBetween(viewerId, profileId)) {
            throw new ForbiddenError('Você não tem permissão para visualizar este perfil');
        }

        if (mustBeFriends) {
            const f = await safeQuery(
                `SELECT 1 FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?) LIMIT 1`,
                [viewerId, profileId, profileId, viewerId]
            );
            if (!f.length) throw new ForbiddenError('Você só pode ver perfis de amigos');
        }

        // Puxa todos os campos do usuário que o front agora exibe
        const rows = await safeQuery(
            `SELECT
         id, username, email, gender, birth_date, height, weight, profile_picture,
         pronouns, occupation, state, city, bio,
         relationship_goals, relationship_type, has_children, drinks, smokes,
         political_orientation, religion,
         kitchen_persona, diet_style, pets, coffee, sports_role, party_style, gangster_persona,
         with_you, perfect_day,           -- JSON (string) ou NULL
         social_links,                    -- JSON (string) ou NULL
         explorer, explorer_type
       FROM users WHERE id=?`,
            [profileId]
        );
        const u = getFirstOrThrow(rows, 'Usuário não encontrado');

        // Interests (slugs) – tabela usada pelo /auth/update-interests
        let interests = [];
        try {
            const intRows = await safeQuery(
                `SELECT i.slug
                   FROM user_interests ui
                   JOIN interests i ON i.id = ui.interest_id
                  WHERE ui.user_id = ?
                  ORDER BY i.label`,
                [profileId]
            );
            interests = intRows.map(r => r.slug);
        } catch { interests = []; }

        // Parse seguro
        const parseJson = (v, fallback) => {
            if (!v) return fallback;
            try { return JSON.parse(v); } catch { return fallback; }
        };

        return {
            id: u.id,
            username: u.username,
            email: u.email || null,
            profile_picture: u.profile_picture || null,
            gender: u.gender || null,
            birth_date: u.birth_date || null,
            height: u.height || null,
            weight: u.weight || null,

            // novos campos
            pronouns: u.pronouns || '',
            occupation: u.occupation || '',
            state: u.state || '',
            city: u.city || '',
            bio: u.bio || '',

            relationship_goals: u.relationship_goals || '',
            relationship_type: u.relationship_type || '',
            has_children: u.has_children || '',
            drinks: u.drinks || '',
            smokes: u.smokes || '',
            political_orientation: u.political_orientation || '',
            religion: u.religion || '',

            kitchen_persona: u.kitchen_persona || '',
            diet_style: u.diet_style || '',
            pets: u.pets || '',
            coffee: u.coffee || '',
            sports_role: u.sports_role || '',
            party_style: u.party_style || '',
            gangster_persona: u.gangster_persona || '',

            with_you: Array.isArray(u.with_you) ? u.with_you : parseJson(u.with_you, []),
            perfect_day: Array.isArray(u.perfect_day) ? u.perfect_day : parseJson(u.perfect_day, []),

            social_links: Array.isArray(u.social_links) ? u.social_links : parseJson(u.social_links, []),

            explorer: !!u.explorer,
            explorer_type: u.explorer_type || 'private',

            interests, // slugs
        };
    },

    async viewExplorerProfile(viewerId, profileId) {
        viewerId = numOrThrow(viewerId); profileId = numOrThrow(profileId);
        if (await isBlockedBetween(viewerId, profileId)) {
            throw new ForbiddenError('Você não tem permissão para visualizar este perfil');
        }

        // Dono do perfil sempre pode ver
        if (viewerId === profileId) {
            return await this.viewProfile(viewerId, profileId, false);
        }

        // Se já são amigos, libera (independe do Explorer)
        const fr = await safeQuery(
            `SELECT 1 FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?) LIMIT 1`,
            [viewerId, profileId, profileId, viewerId]
        );
        if (fr.length) return await this.viewProfile(viewerId, profileId, false);

        // Não são amigos → precisa ter Explorer público
        const rows = await safeQuery(
            `SELECT explorer, explorer_type FROM users WHERE id=? LIMIT 1`,
            [profileId]
        );
        const u = getFirstOrThrow(rows, 'Usuário não encontrado');
        if (!u.explorer) throw new ForbiddenError('Este usuário não está com o Explorer ativo.');
        if (u.explorer_type === 'private') throw new ForbiddenError('Este perfil só é visível para amigos.');

        // Monta payload igual ao viewProfile
        return await this.viewProfile(viewerId, profileId, false);
    },

    // === BLOCK / UNBLOCK / LIST ===
    async blockUser(blockerId, blockedId) {
        blockerId = numOrThrow(blockerId); blockedId = numOrThrow(blockedId);
        if (blockerId === blockedId) throw new BadRequestError('Você não pode se bloquear');

        const exists = await safeQuery(`SELECT id FROM blacklists WHERE blocker_id=? AND blocked_id=?`, [blockerId, blockedId]);
        if (exists.length) throw new ConflictError('Usuário já está bloqueado');

        const conn = await db.getConnection();
        try {
            await conn.beginTransaction();

            await safeQuery(
                `DELETE FROM friendships WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?)`,
                [blockerId, blockedId, blockedId, blockerId], conn
            );
            await safeQuery(
                `DELETE FROM friendship_requests WHERE (sender_id=? AND receiver_id=?) OR (sender_id=? AND receiver_id=?)`,
                [blockerId, blockedId, blockedId, blockerId], conn
            );
            await safeQuery(`INSERT INTO blacklists (blocker_id, blocked_id) VALUES (?,?)`, [blockerId, blockedId], conn);

            await conn.commit();
            return { message: 'Usuário bloqueado com sucesso' };
        } catch (e) {
            await conn.rollback(); throw e;
        } finally { conn.release(); }
    },

    async unblockUser(blockerId, blockedId) {
        blockerId = numOrThrow(blockerId); blockedId = numOrThrow(blockedId);
        const ex = await safeQuery(`SELECT id FROM blacklists WHERE blocker_id=? AND blocked_id=?`, [blockerId, blockedId]);
        if (!ex.length) throw new NotFoundError('Bloqueio não encontrado');

        // ✅ compatível com wrappers que retornam array ou objeto
        const q = await db.query(
            `DELETE FROM blacklists WHERE blocker_id=? AND blocked_id=?`,
            [blockerId, blockedId]
        );
        // Se vier [rows, fields], pegue q[0]; se vier objeto direto, use q.
        const result = Array.isArray(q) ? q[0] : q;
        const affected = Number(
            (result && (result.affectedRows ?? result.affected_rows)) ?? 0
        );
        if (affected <= 0) {
            throw new BadRequestError('Não foi possível remover o bloqueio');
        }
        return { message: 'Usuário desbloqueado com sucesso' };
    },

    async checkIfBlocked(blockerId, blockedId) {
        blockerId = numOrThrow(blockerId); blockedId = numOrThrow(blockedId);
        const r = await safeQuery(`SELECT id FROM blacklists WHERE blocker_id=? AND blocked_id=?`, [blockerId, blockedId]);
        return !!r.length;
    },

    async listBlockedUsers(userId) {
        userId = numOrThrow(userId);
        return await safeQuery(
            `SELECT u.id, u.username, u.profile_picture, b.created_at
             FROM blacklists b
             JOIN users u ON u.id = b.blocked_id
             WHERE b.blocker_id=? ORDER BY b.created_at DESC`,
            [userId]
        );
    },

    // export internal util for service
    isBlockedBetween
};
