// services/chatService.js
const EventEmitter = require('events');
const { phaseDurationSec } = require('../config/appConfig');

class ChatService extends EventEmitter {
    constructor() {
        super();
        this.chatRooms = new Map();
    }

    joinChat(roomId, sessionId, socket, username = 'Anônimo', gender = 'unknown') {
        if (!this.chatRooms.has(roomId)) {
            this.chatRooms.set(roomId, {
                participants: new Map(),
                phases: [],
                currentPhase: 0,
                timer: null,
                pairs: new Map(),
                creationTime: Date.now(),
                category: this.getCategoryFromRoomId(roomId),

                phaseDurationSec,
                phaseEndsAt: null
            });
        }

        const room = this.chatRooms.get(roomId);

        if (room.participants.has(sessionId)) {
            const participant = room.participants.get(sessionId);
            participant.socket = socket;
            participant.socketId = socket.id;
            return { isNew: false, participant };
        } else {
            const newParticipant = {
                socket,
                socketId: socket.id,
                sessionId,
                userInfo: {
                    username,
                    gender,
                    userId: socket.user?.id ? `user_${socket.user.id}` : null
                },
                joinTime: Date.now()
            };
            room.participants.set(sessionId, newParticipant);
            return { isNew: true, participant: newParticipant };
        }
    }

    getCategoryFromRoomId(roomId) {
        if (roomId.includes('heterossexual')) return 'heterossexual';
        if (roomId.includes('homossexual')) return 'homossexual';
        if (roomId.includes('lgbt')) return 'lgbt';
        return 'unknown';
    }

    checkRoomReady(roomId) {
        const room = this.chatRooms.get(roomId);
        if (!room) return false;

        const participants = Array.from(room.participants.values());
        const allReady = participants.every(p => p.userInfo.username && p.userInfo.gender);
        const requiredUsers = this.getMaxUsersForRoom(roomId);

        return participants.length >= requiredUsers && allReady;
    }

    getMaxUsersForRoom(roomId) {
        if (roomId.includes('_sala1')) return 2;
        if (roomId.includes('_sala2')) return 4;
        if (roomId.includes('_sala3')) return 6;
        return 2;
    }

    setupChatRoom(roomId) {
        const room = this.chatRooms.get(roomId);
        const participants = Array.from(room.participants.values());

        if (room.category === 'heterossexual') {
            const men = participants.filter(p => p.userInfo.gender === 'male');
            const women = participants.filter(p => p.userInfo.gender === 'female');

            if (men.length !== women.length) {
                throw new Error('Número desigual de homens e mulheres em sala heterossexual');
            }
            this.setupHeterosexualPhases(roomId, men, women);
        } else if (room.category === 'homossexual') {
            const firstGender = participants[0]?.userInfo.gender;
            if (!participants.every(p => p.userInfo.gender === firstGender)) {
                throw new Error('Participantes de gêneros diferentes em sala homossexual');
            }
            this.setupHomosexualPhases(roomId, participants);
        } else {
            this.setupGenericPhases(roomId, participants);
        }

        return room;
    }

    setupHeterosexualPhases(roomId, men, women) {
        const room = this.chatRooms.get(roomId);
        const numPairs = men.length;
        room.phases = [];

        if (numPairs === 1) {
            room.phases.push({ pairs: [[men[0], women[0]]] });
        } else if (numPairs === 2) {
            room.phases.push({ pairs: [[men[0], women[0]], [men[1], women[1]]] });
            room.phases.push({ pairs: [[men[0], women[1]], [men[1], women[0]]] });
        } else if (numPairs === 3) {
            room.phases.push({ pairs: [[men[0], women[0]], [men[1], women[1]], [men[2], women[2]]] });
            room.phases.push({ pairs: [[men[0], women[1]], [men[1], women[2]], [men[2], women[0]]] });
            room.phases.push({ pairs: [[men[0], women[2]], [men[1], women[0]], [men[2], women[1]]] });
        }

        room.totalPhases = room.phases.length;
        return room;
    }

    setupHomosexualPhases(roomId, participants) {
        const room = this.chatRooms.get(roomId);
        const numParticipants = participants.length;
        room.phases = [];

        if (numParticipants === 2) {
            room.phases.push({ pairs: [[participants[0], participants[1]]] });
        } else if (numParticipants === 4) {
            room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]]] });
            room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[3]]] });
        } else if (numParticipants === 6) {
            room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]], [participants[4], participants[5]]] });
            room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[4]], [participants[3], participants[5]]] });
            room.phases.push({ pairs: [[participants[0], participants[3]], [participants[1], participants[5]], [participants[2], participants[4]]] });
        }

        room.totalPhases = room.phases.length;
        return room;
    }

    setupGenericPhases(roomId, participants) {
        const room = this.chatRooms.get(roomId);
        const numParticipants = participants.length;
        room.phases = [];

        if (numParticipants === 2) {
            room.phases.push({ pairs: [[participants[0], participants[1]]] });
        } else if (numParticipants === 4) {
            room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]]] });
            room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[3]]] });
        } else if (numParticipants === 6) {
            room.phases.push({ pairs: [[participants[0], participants[1]], [participants[2], participants[3]], [participants[4], participants[5]]] });
            room.phases.push({ pairs: [[participants[0], participants[2]], [participants[1], participants[4]], [participants[3], participants[5]]] });
            room.phases.push({ pairs: [[participants[0], participants[3]], [participants[1], participants[5]], [participants[2], participants[4]]] });
        }

        room.totalPhases = room.phases.length;
        return room;
    }

    startNextPhase(roomId) {
        const room = this.chatRooms.get(roomId);
        if (!room) return;

        if (room.timer) clearInterval(room.timer);
        if (room.currentPhase >= room.phases.length) {
            this.emit('endChat', roomId);
            return;
        }

        this.emit('clearChatHistory', roomId);
        this.emit('phaseTransition', roomId, { message: 'Preparando próximo parceiro...' });

        setTimeout(() => {
            const currentPhase = room.phases[room.currentPhase];
            room.pairs.clear();

            currentPhase.pairs.forEach(pair => {
                const [user1, user2] = pair;
                if (!user1 || !user2) {
                    console.error('Par inválido encontrado:', pair);
                    return;
                }

                room.pairs.set(user1.socketId, user2.socketId);
                room.pairs.set(user2.socketId, user1.socketId);
            });

            const phaseData = {
                phase: room.currentPhase + 1,
                totalPhases: room.totalPhases,
                pairs: room.pairs,
                phaseDurationSec: room.phaseDurationSec
            };

            this.emit('phaseStarted', roomId, phaseData);

            room.currentPhase++;

            let timeLeft = room.phaseDurationSec;
            room.phaseEndsAt = Date.now() + (timeLeft * 1000);
            this.emit('timeUpdate', roomId, { timeLeft });

            room.timer = setInterval(() => {
                const remaining = Math.max(0, Math.ceil((room.phaseEndsAt - Date.now()) / 1000));
                this.emit('timeUpdate', roomId, { timeLeft: remaining });
                if (remaining <= 0) {
                    clearInterval(room.timer);
                    room.timer = null;
                    room.phaseEndsAt = null;
                    this.startNextPhase(roomId);
                }
            }, 1000);
        }, 2000);
    }

    extractUserIdFromSession(participant) {
        return participant.userId ||
            (participant.userInfo && participant.userInfo.userId) ||
            null;
    }

    async prepareEndChatData(roomId) {
        const room = this.chatRooms.get(roomId);
        if (!room) return null;

        const endTime = new Date();
        const results = [];

        for (const participant of room.participants.values()) {
            const partners = new Set();

            for (const phase of room.phases) {
                for (const pair of phase.pairs) {
                    if (pair.some(p => p.sessionId === participant.sessionId)) {
                        pair.forEach(p => {
                            if (p.sessionId !== participant.sessionId) {
                                partners.add({
                                    userId: p.userInfo.userId || this.extractUserIdFromSession(p),
                                    username: p.userInfo.username,
                                    gender: p.userInfo.gender,
                                    sessionId: p.sessionId
                                });
                            }
                        });
                    }
                }
            }

            results.push({
                currentUser: {
                    userId: participant.userInfo.userId || this.extractUserIdFromSession(participant),
                    username: participant.userInfo.username,
                    gender: participant.userInfo.gender,
                    sessionId: participant.sessionId
                },
                partners: Array.from(partners),
                roomId: roomId,
                endTime: endTime.toISOString()
            });
        }

        return results;
    }

    handleMessage(roomId, socketId, message) {
        const room = this.chatRooms.get(roomId);
        if (!room) throw new Error('Sala não encontrada');
        
        const partnerId = room.pairs.get(socketId);
        if (!partnerId) throw new Error('Nenhum parceiro definido');

        return {
            partnerId,
            messageData: {
                sender: socketId,
                message,
                timestamp: new Date().toISOString(),
                isSelf: false
            }
        };
    }

    handleDisconnect(socketId) {
        const disconnectedData = {};

        this.chatRooms.forEach((room, roomId) => {
            let disconnectedUser = null;

            for (const [sessionId, participant] of room.participants) {
                if (participant.socketId === socketId) {
                    disconnectedUser = participant;
                    break;
                }
            }

            if (disconnectedUser) {
                const partnerId = room.pairs.get(socketId);
                disconnectedUser.socket = null;
                disconnectedUser.socketId = null;

                disconnectedData[roomId] = {
                    partnerId,
                    disconnectedUser,
                    room
                };
            }
        });

        return disconnectedData;
    }

    cleanupDisconnectedUser(roomId, sessionId) {
        const room = this.chatRooms.get(roomId);
        if (!room) return;

        const participant = room.participants.get(sessionId);
        if (participant && !participant.socketId) {
            room.participants.delete(sessionId);
            room.pairs.delete(participant.socketId);
            return true;
        }
        return false;
    }

    updateUserData(roomId, sessionId, userData) {
        const room = this.chatRooms.get(roomId);
        if (!room) throw new Error('Sala não encontrada');
        
        const participant = room.participants.get(sessionId);
        if (participant) {
            participant.userInfo = {
                username: userData.username,
                gender: userData.gender
            };
            return true;
        }
        return false;
    }

    getPartnerInfo(roomId, socketId) {
        const room = this.chatRooms.get(roomId);
        if (!room) return null;

        const partnerSocketId = room.pairs.get(socketId);
        if (!partnerSocketId) return null;

        const partner = Array.from(room.participants.values())
            .find(p => p.socketId === partnerSocketId);

        if (!partner) return null;

        return {
            username: partner.userInfo.username,
            gender: partner.userInfo.gender,
            sessionId: partner.sessionId,
            socketId: partner.socketId,
            userId: partner.userInfo.userId || this.extractUserIdFromSession(partner)
        };
    }
}

module.exports = ChatService;
