// C:\src\Clientes\Datetime\services\authService.js
const User = require('../models/users');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const path = require('path');
const fs = require('fs');
const tokenStore = require('../models/tokenStore');
const { AppError, NotFoundError, ForbiddenError, ConflictError, BadRequestError } = require('../config/errors');

const jwtSecret = process.env.JWT_SECRET;
const jwtRefreshSecret = process.env.JWT_REFRESH_SECRET;
if (!jwtSecret || !jwtRefreshSecret) {
  throw new Error('JWT_SECRET e JWT_REFRESH_SECRET devem estar configurados no ambiente.');
}

const saltRounds = Number(process.env.BCRYPT_ROUNDS || 12);
const ACCESS_TTL = process.env.ACCESS_TTL || '15m'; // 15 minutos
const REFRESH_TTL_DAYS = Number(process.env.REFRESH_TTL_DAYS || 7); // 7 dias

function addDays(date, days) {
  const d = new Date(date);
  d.setDate(d.getDate() + days);
  return d;
}

// ===== helpers =====

// Normaliza caminhos que começam com '/uploads/...' para 'uploads/...'
function stripLeadingSlash(p) {
  return typeof p === 'string' ? p.replace(/^\//, '') : p;
}

/** Remove QUALQUER arquivo sob /public (recebe algo tipo '/uploads/11/profile/x.png') */
async function removePublicFile(filePath) {
  const rel = stripLeadingSlash(filePath); // 'uploads/11/profile/x.png'
  const fullPath = path.join(__dirname, '../public', rel);
  try {
    if (fs.existsSync(fullPath)) fs.unlinkSync(fullPath);
  } catch (e) {
    // não quebrar o fluxo se falhar aqui
  }
}

function normalizeSocialLinks(raw) {
  if (!Array.isArray(raw)) return [];

  const trim = (s) => String(s || '').trim();

  // remove domínio conhecido e @handle quando fizer sentido
  const mapBase = {
    'Instagram': (v) => trim(v).replace(/^@/, '').replace(/^https?:\/\/(www\.)?instagram\.com\//i, ''),
    'X/Twitter': (v) => trim(v).replace(/^@/, '').replace(/^https?:\/\/(www\.)?(twitter|x)\.com\//i, ''),
    'LinkedIn': (v) => trim(v).replace(/^https?:\/\/(www\.)?linkedin\.com\//i, ''), // vira "in/user" | "company/x"
    'Facebook': (v) => trim(v).replace(/^https?:\/\/(www\.)?facebook\.com\//i, ''),
    'TikTok': (v) => trim(v).replace(/^@/, '').replace(/^https?:\/\/(www\.)?tiktok\.com\//i, ''),
    'YouTube': (v) => trim(v).replace(/^https?:\/\/(www\.)?youtube\.com\//i, ''),
    'Site/Portfólio': (v) => trim(v),
    'Outro': (v) => trim(v),
  };

  // montar URL final quando veio "handle" (sem http)
  const toUrl = {
    'Instagram': (h) => `https://instagram.com/${h.replace(/^\/+/, '')}`,
    'X/Twitter': (h) => `https://x.com/${h.replace(/^\/+/, '')}`,
    'TikTok': (h) => `https://tiktok.com/@${h.replace(/^@+/, '').replace(/^\/+/, '')}`,
    'LinkedIn': (h) => `https://www.linkedin.com/${h.replace(/^\/+/, '')}`, // aceita "in/fulano", "company/org", etc.
    'Facebook': (h) => `https://www.facebook.com/${h.replace(/^\/+/, '')}`,
    'YouTube': (h) => `https://www.youtube.com/${h.replace(/^\/+/, '')}`,
    'Site/Portfólio': (h) => /^https?:\/\//i.test(h) ? h : `https://${h}`,
    'Outro': (h) => /^https?:\/\//i.test(h) ? h : `https://${h}`,
  };

  const out = [];
  for (const item of raw) {
    if (!item || !item.network) continue;
    const network = String(item.network).trim();
    let url = trim(item.url);
    if (!url) continue;

    const normalizer = mapBase[network] || ((v) => trim(v));
    let handle = normalizer(url);

    // Se já parece URL completa, mantém; senão, usa builder por rede (ou fallback https://)
    if (/^https?:\/\//i.test(handle)) {
      url = handle;
    } else {
      const make = toUrl[network] || toUrl['Site/Portfólio'];
      url = make(handle);
    }

    // Validação final: descarta URLs obviamente inválidas
    try {
      // lança se inválida
      new URL(url);
      out.push({ network, url });
    } catch {
      // ignora item malformado
    }
  }
  return out;
}

const normStr = (v) => v === '' ? null : v;
// helper p/ números: '' -> null, '170' -> 170, undefined -> undefined
const normNum = (v) => v === '' || v === undefined ? null : Number(v);

const ENUMS = {
  pronouns: ['ela/dela', 'ele/dele', 'elu/delu'],

  relationship_goals: ['amizade', 'casual', 'namoro', 'casamento', 'networking', 'parceria', 'nao_sei'],
  relationship_type: ['monogamico', 'aberto', 'poliamor', 'discreto', 'nao_sei'],

  has_children: [
    'tenho_nao_quero_mais', 'tenho', 'tenho_quero_mais',
    'nao_tenho_nao_quero', 'nao_tenho_aberto', 'prefiro_nao_dizer'
  ],
  drinks: ['nao', 'socialmente', 'raramente', 'frequentemente'],
  smokes: ['nao', 'ocasional', 'vape', 'sim'],
  political_orientation: [
    'esquerda', 'centro_esquerda', 'centro', 'centro_direita', 'direita', 'libertario', 'independente', 'prefiro_nao_dizer'
  ],
  religion: [
    'catolico', 'evangelico', 'espirita', 'umbanda_candomble', 'judaico', 'muculmano', 'budista', 'agnostico', 'ateu', 'outra', 'prefiro_nao_dizer'
  ],
  kitchen_persona: [
    'chef_fim_de_semana', 'mestre_microondas', 'rei_da_airfryer', 'tempero_da_vovo', 'queima_ate_gelo'
  ],
  diet_style: [
    'conto_macros', 'low_carb', 'vegetariano', 'vegano', 'onivoro_feliz', 'segunda_sem_carne', 'jejum_intermitente', 'nao_faco_ideia'
  ],
  pets: ['team_dogs', 'team_cats', 'amo_todos', 'alergia_mas_tento', 'peixes_e_plantas', 'prefiro_plantas'],
  coffee: ['sempre', 'com_acucar', 'com_leite', 'descafeinado', 'cha', 'nao_curto'],
  sports_role: ['atleta_dedicado', 'torcedor_de_sofa', 'domingo_no_parque', 'marombeiro', 'yoga_pilates', 'corrida', 'ciclista'],
  party_style: [
    'dj_improvisado', 'guardiao_da_bolsa', 'dono_da_roda', 'primeiro_na_pista', 'volta_com_batata', 'fotografo', 'deus_me_livre', 'prefiro_nem_ir'
  ],
  gangster_persona: ['estrategista', 'motorista_fuga', 'hacker', 'pacificador', 'esquece_senha']
};

const allowOrNull = (group, val) => {
  if (val === undefined || val === '') return null;
  const v = String(val).trim().toLowerCase();
  return ENUMS[group]?.includes(v) ? v : null;
};

class AuthService {
  // ===== tokens =====
  generateAccessToken(user) {
    return jwt.sign({ sub: String(user.id) }, jwtSecret, { expiresIn: ACCESS_TTL });
  }
  generateRefreshToken(user) {
    return jwt.sign({ sub: String(user.id) }, jwtRefreshSecret, { expiresIn: `${REFRESH_TTL_DAYS}d` });
  }
  verifyAccess(token) { return jwt.verify(token, jwtSecret); }
  verifyRefresh(token) { return jwt.verify(token, jwtRefreshSecret); }

  // ===== login =====
  async login(email, password, ctx = {}) {
    const user = await User.findByEmail(email);
    if (!user) throw new NotFoundError('Credenciais inválidas');

    const validPassword = await bcrypt.compare(password, user.password);
    if (!validPassword) throw new ForbiddenError('Credenciais inválidas');

    const accessToken = this.generateAccessToken(user);
    const refreshToken = this.generateRefreshToken(user);

    const expiresAt = addDays(new Date(), REFRESH_TTL_DAYS);
    await tokenStore.saveRefreshToken({
      userId: user.id,
      refreshToken,
      expiresAt,
      ip: ctx.ip,
      userAgent: ctx.userAgent
    });

    return {
      accessToken,
      refreshToken,
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        profilePicture: user.profile_picture,
        gender: user.gender,
        height: user.height,
        weight: user.weight,
        explorer: user.explorer,
        explorer_type: user.explorer_type,
        created_at: user.created_at
      }
    };
  }

  // ===== refresh com rotação =====
  async refresh(oldRefreshToken, ctx = {}) {
    if (!oldRefreshToken) throw new BadRequestError('Refresh token não fornecido');

    let decoded;
    try {
      decoded = this.verifyRefresh(oldRefreshToken);
    } catch {
      throw new ForbiddenError('Refresh token inválido');
    }

    const userId = Number(decoded.sub);

    const stored = await tokenStore.findByToken(oldRefreshToken);
    if (!stored) throw new ForbiddenError('Refresh token não reconhecido');

    if (stored.revoked_at) {
      await tokenStore.revokeAllForUser(userId); // opcional, força logout de tudo em replay
      throw new ForbiddenError('Refresh token já utilizado (revogado).');
    }

    if (new Date(stored.expires_at) < new Date()) {
      await tokenStore.markRevokedByHash(stored.token_hash);
      throw new ForbiddenError('Refresh token expirado');
    }

    // rotação
    const accessToken = this.generateAccessToken({ id: userId });
    const refreshToken = this.generateRefreshToken({ id: userId });

    const newExpiresAt = addDays(new Date(), REFRESH_TTL_DAYS);
    const newHash = await tokenStore.saveRefreshToken({
      userId,
      refreshToken,
      expiresAt: newExpiresAt,
      ip: ctx.ip,
      userAgent: ctx.userAgent
    });

    await tokenStore.markRevokedByHash(stored.token_hash, newHash);

    return { accessToken, refreshToken };
  }

  async logout(refreshToken) {
    if (refreshToken) {
      const stored = await tokenStore.findByToken(refreshToken);
      if (stored && !stored.revoked_at) {
        await tokenStore.markRevokedByHash(stored.token_hash);
      }
    }
  }

  async logoutAll(userId) {
    await tokenStore.revokeAllForUser(userId);
  }

  // ===== user data / profile / gallery =====
  async register(username, email, password, gender) {
    const existingUser = await User.findByEmail(email);
    if (existingUser) throw new ConflictError('Email já cadastrado');

    const hashedPassword = await bcrypt.hash(password, saltRounds);
    const userId = await User.create({ username, email, password: hashedPassword, gender });
    return userId;
  }

  async getUserData(token) {
    if (!token) throw new BadRequestError('Token não fornecido');
    const decoded = this.verifyAccess(token);
    const user = await User.findById(decoded.sub || decoded.id);
    if (!user) throw new NotFoundError('Usuário não encontrado');

    const interests = await User.getUserInterests(user.id);
    let social_links = null, with_you = null, perfect_day = null;
    try { social_links = user.social_links ? JSON.parse(user.social_links) : null; } catch { }
    try { with_you = user.with_you ? JSON.parse(user.with_you) : null; } catch { }
    try { perfect_day = user.perfect_day ? JSON.parse(user.perfect_day) : null; } catch { }

    return {
      id: user.id, username: user.username, email: user.email, gender: user.gender,
      birthDate: user.birth_date, height: user.height, weight: user.weight,
      profilePicture: user.profile_picture, explorer: user.explorer, explorer_type: user.explorer_type,

      pronouns: user.pronouns, bio: user.bio, occupation: user.occupation, city: user.city, state: user.state,
      relationship_goals: user.relationship_goals, relationship_type: user.relationship_type, has_children: user.has_children,
      drinks: user.drinks, smokes: user.smokes, political_orientation: user.political_orientation, religion: user.religion,

      kitchen_persona: user.kitchen_persona, diet_style: user.diet_style, pets: user.pets, coffee: user.coffee,
      sports_role: user.sports_role, party_style: user.party_style, gangster_persona: user.gangster_persona,

      social_links, with_you, perfect_day,
      interests_labels: interests.map(i => i.label), // útil para render
      interests: interests.map(i => i.slug)         // útil para salvar/estado
    };
  }

  async updateProfile(token, updateData, file) {
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);

    // Sanitiza & normaliza
    const processed = {
      username: normStr(updateData.username),

      birthDate: updateData.birthDate === '' ? null : updateData.birthDate,

      height: normNum(updateData.height),
      weight: normNum(updateData.weight),

      pronouns: allowOrNull('pronouns', updateData.pronouns),
      occupation: normStr(updateData.occupation),
      city: normStr(updateData.city),
      state: normStr(updateData.state),
      bio: normStr(updateData.bio),

      relationship_goals: allowOrNull('relationship_goals', updateData.relationship_goals),
      relationship_type: allowOrNull('relationship_type', updateData.relationship_type),
      has_children: allowOrNull('has_children', updateData.has_children),
      drinks: allowOrNull('drinks', updateData.drinks),
      smokes: allowOrNull('smokes', updateData.smokes),
      political_orientation: allowOrNull('political_orientation', updateData.political_orientation),
      religion: allowOrNull('religion', updateData.religion),

      kitchen_persona: allowOrNull('kitchen_persona', updateData.kitchen_persona),
      diet_style: allowOrNull('diet_style', updateData.diet_style),
      pets: allowOrNull('pets', updateData.pets),
      coffee: allowOrNull('coffee', updateData.coffee),
      sports_role: allowOrNull('sports_role', updateData.sports_role),
      party_style: allowOrNull('party_style', updateData.party_style),
      gangster_persona: allowOrNull('gangster_persona', updateData.gangster_persona)
    };



    // JSON strings vindas do multipart
    const parseMaybeJson = (s) => { try { return s ? JSON.parse(s) : null; } catch { return null; } };

    const rawSocial = parseMaybeJson(updateData.social_links);
    processed.social_links = normalizeSocialLinks(rawSocial);

    processed.with_you = parseMaybeJson(updateData.with_you) ?? [];
    processed.perfect_day = parseMaybeJson(updateData.perfect_day) ?? [];

    if (file) {
      const current = await User.findById(userId);
      if (current && current.profile_picture) {
        await removePublicFile(current.profile_picture); // <<< troquei aqui
      }
      processed.profilePicture = `/uploads/${userId}/profile/${file.filename}`;
    }

    await User.update(userId, processed);
    // ⚠️ Retorna snapshot consistente (mesmo shape do /auth/me)
    const fresh = await User.findById(userId);
    const interests = await User.getUserInterests(userId);
    let social_links = null, with_you = null, perfect_day = null;
    try { social_links = fresh.social_links ? JSON.parse(fresh.social_links) : null; } catch { }
    try { with_you = fresh.with_you ? JSON.parse(fresh.with_you) : null; } catch { }
    try { perfect_day = fresh.perfect_day ? JSON.parse(fresh.perfect_day) : null; } catch { }
    return {
      id: fresh.id,
      username: fresh.username,
      email: fresh.email,
      gender: fresh.gender,
      birthDate: fresh.birth_date,
      height: fresh.height,
      weight: fresh.weight,
      profilePicture: fresh.profile_picture,
      explorer: fresh.explorer,
      explorer_type: fresh.explorer_type,
      pronouns: fresh.pronouns, bio: fresh.bio, occupation: fresh.occupation, city: fresh.city, state: fresh.state,
      relationship_goals: fresh.relationship_goals, relationship_type: fresh.relationship_type, has_children: fresh.has_children,
      drinks: fresh.drinks, smokes: fresh.smokes, political_orientation: fresh.political_orientation, religion: fresh.religion,
      kitchen_persona: fresh.kitchen_persona, diet_style: fresh.diet_style, pets: fresh.pets, coffee: fresh.coffee,
      sports_role: fresh.sports_role, party_style: fresh.party_style, gangster_persona: fresh.gangster_persona,
      social_links, with_you, perfect_day,
      interests_labels: interests.map(i => i.label),
      interests: interests.map(i => i.slug)
    };
  }

  async updateUserInterests(token, slugs) {
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);
    await User.setUserInterests(userId, Array.isArray(slugs) ? slugs : []);
    return true;
  }

  /**
* Calcula % de conclusão do perfil e aponta campos que faltam.
* Retorna { percent, missing[], counts, weights }.
*/
  async profileCompletion(token) {
    if (!token) throw new BadRequestError('Token não fornecido');
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);

    const u = await User.findById(userId);
    if (!u) throw new NotFoundError('Usuário não encontrado');

    // Parse de JSONs que podem contribuir
    let social_links = null, with_you = null, perfect_day = null;
    try { social_links = u.social_links ? JSON.parse(u.social_links) : null; } catch { }
    try { with_you = u.with_you ? JSON.parse(u.with_you) : null; } catch { }
    try { perfect_day = u.perfect_day ? JSON.parse(u.perfect_day) : null; } catch { }

    // Interesses e galeria
    const interests = await User.getUserInterests(userId);        // array [{slug,label}]
    const gallery = await User.getGalleryImages(userId);         // array

    // Helpers
    const has = (v) => v !== null && v !== undefined && String(v).trim() !== '';
    const hasNum = (v) => v !== null && v !== undefined && !Number.isNaN(Number(v)) && Number(v) > 0;
    const count = (arr) => Array.isArray(arr) ? arr.length : 0;

    // ===== REGRA DE PONTUAÇÃO (fecha em 100) =====
    // Essenciais (50)
    const rules = [
      { key: 'username', label: 'Nome de usuário', weight: 5, ok: has(u.username) },
      { key: 'profile_picture', label: 'Foto de perfil', weight: 10, ok: has(u.profile_picture) },
      { key: 'birth_date', label: 'Data de nascimento', weight: 10, ok: has(u.birth_date) },
      { key: 'city_state', label: 'Cidade/Estado', weight: 10, ok: has(u.city) && has(u.state) },
      { key: 'bio', label: 'Bio', weight: 10, ok: has(u.bio) },
      { key: 'occupation', label: 'Ocupação', weight: 5, ok: has(u.occupation) },

      // Básicos (11)
      { key: 'gender', label: 'Gênero', weight: 5, ok: has(u.gender) },
      { key: 'height', label: 'Altura', weight: 3, ok: hasNum(u.height) },
      { key: 'weight', label: 'Peso', weight: 3, ok: hasNum(u.weight) },

      // Preferências (21) – 7 itens × 3
      { key: 'relationship_goals', label: 'Objetivo de relacionamento', weight: 3, ok: has(u.relationship_goals) },
      { key: 'relationship_type', label: 'Tipo de relacionamento', weight: 3, ok: has(u.relationship_type) },
      { key: 'has_children', label: 'Filhos', weight: 3, ok: has(u.has_children) },
      { key: 'drinks', label: 'Bebidas alcoólicas', weight: 3, ok: has(u.drinks) },
      { key: 'smokes', label: 'Tabagismo', weight: 3, ok: has(u.smokes) },
      { key: 'political_orientation', label: 'Orientação política', weight: 3, ok: has(u.political_orientation) },
      { key: 'religion', label: 'Religião', weight: 3, ok: has(u.religion) },

      // Sociais & Interesses (14)
      { key: 'interests', label: 'Interesses (≥ 3)', weight: 10, ok: count(interests) >= 3 },
      { key: 'social_links', label: 'Link social (≥ 1)', weight: 4, ok: count(social_links) >= 1 },

      // Extras (4)
      { key: 'gallery', label: 'Galeria (≥ 3 fotos)', weight: 2, ok: count(gallery) >= 3 },
      { key: 'with_you', label: 'Comigo você vai… (≥ 2)', weight: 1, ok: count(with_you) >= 2 },
      { key: 'perfect_day', label: 'Meu dia perfeito (≥ 2)', weight: 1, ok: count(perfect_day) >= 2 },
    ];

    const total = rules.reduce((s, r) => s + r.weight, 0);   // deve dar 100
    const earned = rules.reduce((s, r) => s + (r.ok ? r.weight : 0), 0);
    const percent = Math.max(0, Math.min(100, Math.round((earned / total) * 100)));

    // Lista do que falta (com dicas)
    const hints = {
      username: 'Defina um nome que as pessoas verão nas salas.',
      profile_picture: 'Envie uma foto nítida do rosto.',
      birth_date: 'Informe sua data de nascimento para exibirmos sua idade.',
      city_state: 'Preencha cidade e estado para conexões locais.',
      bio: 'Escreva um breve resumo sobre você.',
      occupation: 'Conte com o que você trabalha ou estuda.',
      gender: 'Selecione seu gênero.',
      height: 'Informe sua altura (cm).',
      weight: 'Informe seu peso (kg).',
      relationship_goals: 'Escolha o que está buscando.',
      relationship_type: 'Selecione o tipo de relação.',
      has_children: 'Informe se tem filhos.',
      drinks: 'Conte se você bebe e com que frequência.',
      smokes: 'Conte se fuma.',
      political_orientation: 'Selecione sua orientação.',
      religion: 'Informe sua religião.',
      interests: 'Adicione ao menos 3 interesses.',
      social_links: 'Inclua pelo menos um link social.',
      gallery: 'Envie pelo menos 3 fotos na galeria.',
      with_you: 'Adicione 2 ou mais itens em “Comigo você vai…”.',
      perfect_day: 'Adicione 2 ou mais itens em “Dia perfeito”.'
    };

    const missing = rules.filter(r => !r.ok).map(r => ({
      key: r.key,
      label: r.label,
      hint: hints[r.key] || ''
    }));

    return {
      percent,
      missing,
      counts: { done: rules.length - missing.length, missing: missing.length, total: rules.length },
      weights: { total, earned }
    };
  }



  async removeProfilePicture(token) {
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);
    const user = await User.findById(userId);
    if (user && user.profile_picture) {
      await removePublicFile(user.profile_picture); // <<< troquei aqui
    }
    await User.updateImage(userId, { profilePicture: null });
  }

  async uploadGalleryImage(token, file) {
    if (!file) throw new BadRequestError('Nenhuma imagem enviada');
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);

    const imagePath = `/uploads/${userId}/gallery/${file.filename}`;
    const imageId = await User.addGalleryImage(userId, imagePath);
    return { id: imageId, path: imagePath };
  }

  async getGalleryImages(token) {
    const decoded = this.verifyAccess(token);
    return User.getGalleryImages(Number(decoded.sub || decoded.id));
  }

  async deleteGalleryImage(token, imageId) {
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);

    const imagePath = await User.deleteGalleryImage(userId, imageId);
    await this.removeProfilePictureFile(imagePath);
  }

  // ===== helpers =====
  async removeProfilePictureFile(filePath) {
    const fullPath = path.join(__dirname, '../public', filePath);
    if (fs.existsSync(fullPath)) fs.unlinkSync(fullPath);
  }

  async listInterests() {
    return await User.listInterests();
  }

  async uploadGalleryImages(token, files) {
    if (!files || !files.length) throw new BadRequestError('Nenhuma imagem enviada');
    const decoded = this.verifyAccess(token);
    const userId = Number(decoded.sub || decoded.id);

    const out = [];
    for (const f of files) {
      const imagePath = `/uploads/${userId}/gallery/${f.filename}`;
      const imageId = await User.addGalleryImage(userId, imagePath);
      out.push({ id: imageId, path: imagePath });
    }
    return out;
  }

}

module.exports = new AuthService();
