// C:\src\Clientes\Datetime\server.js
require('dotenv').config();
const http = require('http');
const socketIo = require('socket.io');
const cookie = require('cookie');
const jwt = require('jsonwebtoken');

const app = require('./app');
const lobbyController = require('./controllers/lobbyController');
const chatController = require('./controllers/chatController');
const chatFriendController = require('./controllers/chatFriendsController');

const User = require('./models/users');

function normalizeGender(g) {
  if (!g) return 'other';
  const s = String(g).trim().toLowerCase();
  const maleSet = new Set(['male','m','masculino','homem','masc']);
  const femaleSet = new Set(['female','f','feminino','mulher','fem']);
  if (maleSet.has(s)) return 'male';
  if (femaleSet.has(s)) return 'female';
  return 'other';
}

const server = http.createServer(app);

const ORIGINS = (process.env.CORS_ORIGINS || 'http://localhost:3001')
  .split(',')
  .map(s => s.trim());

const io = socketIo(server, {
  cors: {
    origin: ORIGINS,
    methods: ['GET', 'POST'],
    credentials: true
  },
  connectionStateRecovery: {
    maxDisconnectionDuration: 60_000,
    skipMiddlewares: false
  },
  pingInterval: 25_000,
  pingTimeout: 60_000
});

// ---- Helper: extrai access token do handshake
function getSocketAccessToken(handshake) {
  const fromAuth = handshake.auth?.token;
  const cookieHeader = handshake.headers?.cookie || '';
  const parsed = cookie.parse(cookieHeader);
  const fromCookie = parsed?.token;
  return fromAuth || fromCookie || null;
}

// ---- Auth do socket + carrega perfil do DB
io.use(async (socket, next) => {
  try {
    const jwtSecret = process.env.JWT_SECRET;
    if (!jwtSecret) return next(new Error('Configuração ausente: JWT_SECRET'));

    const token = getSocketAccessToken(socket.handshake);
    if (!token) return next(new Error('Autenticação requerida'));

    const decoded = jwt.verify(token, jwtSecret);
    const userId = decoded.sub || decoded.id;
    if (!userId) return next(new Error('Token inválido (sem sub/id)'));

    const user = await User.findById(Number(userId));
    if (!user) return next(new Error('USER_NOT_FOUND'));

    socket.user = {
      id: Number(user.id),
      username: user.username || decoded.username || null,
      gender: normalizeGender(user.gender || decoded.gender || null)
    };

    return next();
  } catch (err) {
    if (err.name === 'TokenExpiredError') {
      return next(new Error('TOKEN_EXPIRED'));
    }
    return next(new Error('TOKEN_INVALID'));
  }
});

io.on('connection', (socket) => {
  socket.emit('auth:ok', { userId: socket.user.id });
});

// Inicialização dos controladores
lobbyController.initialize(io);
chatController.initialize(io);
chatFriendController.initialize(io);

const PORT = process.env.PORT || 3001;
server.listen(PORT, () => {
  console.log(`Servidor rodando na porta ${PORT}`);
});

process.on('unhandledRejection', (err) => {
  console.error('Erro não tratado:', err);
  process.exit(1);
});
