const express = require('express');
const router = express.Router();
const path = require('path');
const chatController = require('../controllers/chatController');

// Middleware para configurar headers comuns
const setNoCacheHeaders = (req, res, next) => {
    res.set({
        'Cache-Control': 'no-store, no-cache, must-revalidate, private',
        'Pragma': 'no-cache',
        'Expires': '0'
    });
    next();
};

// Middleware de validação simplificado
const validateChatParams = (req, res, next) => {
    // Verifica se veio do redirecionamento do lobby (com dados no sessionStorage)
    if (req.query.roomId) {
        return next();
    }
    
    // Formato tradicional com dados codificados
    if (req.query.data) {
        try {
            const data = JSON.parse(decodeURIComponent(req.query.data));
            if (!data.roomId || !data.currentUser?.sessionId) {
                throw new Error('Dados incompletos');
            }
            req.chatData = data;
            return next();
        } catch (error) {
            console.error('Erro ao validar parâmetros do chat:', error);
            return res.redirect('/?error=invalid_chat_data');
        }
    }
    
    return res.redirect('/?error=missing_chat_params');
};

// No arquivo routes/chat.js
router.get('/', setNoCacheHeaders, (req, res) => {
    const { roomId, sessionId } = req.query;
    
    if (!roomId || !sessionId) {
        return res.redirect('/?error=missing_params');
    }
    
    res.sendFile(path.join(__dirname, '../views/chat.html'));
});

// Rota alternativa de entrada (HTTP)
router.get('/enter', setNoCacheHeaders, (req, res) => {
    const { roomId, username, gender } = req.query;
    
    if (!roomId || !username || !gender) {
        return res.status(400).json({ 
            success: false,
            error: 'Parâmetros inválidos',
            required: ['roomId', 'username', 'gender']
        });
    }

    const sessionId = chatController.generateSessionId();
    const encodedData = encodeURIComponent(JSON.stringify({
        roomId,
        currentUser: {
            sessionId,
            username,
            gender
        }
    }));
    
    res.redirect(`/chat?data=${encodedData}`);
});

module.exports = router;