document.addEventListener('DOMContentLoaded', initResults);

async function initResults() {
  const wrap = qs('#listWrap');
  const alertBox = qs('#errorMessage');
  const meta = qs('#sessionMeta');
  const btnAddAll = qs('#btnAddAll');

  function showError(msg) {
    alertBox.textContent = msg;
    alertBox.hidden = !msg;
  }

  // 1) Ler dados salvos pelo chat (sessionStorage tem prioridade)
  const chatData = readChatData();
  if (!chatData) {
    wrap.innerHTML = emptyState();
    showError('Não encontrei os dados desta sessão de chat. Tente voltar e entrar novamente.');
    return;
  }

  // 2) Deduplicar parceiros e garantir exclusão do próprio usuário
  const currentId = chatData?.currentUser?.userId || chatData?.currentUser?.sessionId || chatData?.currentUser?.id;
  const uniq = dedupePartners(chatData?.partners || [], currentId);

  // 3) Cabeçalho: data/hora do fim do chat
  const end = parseDate(chatData.endTime);
  if (end) {
    const d = formatDate(end);
    const t = formatTime(end);
    meta.textContent = `Sessão encerrada em ${d} às ${t}`;
    meta.hidden = false;
  }

  // 4) Render
  if (uniq.length === 0) {
    wrap.innerHTML = emptyState();
    btnAddAll.disabled = true;
    return;
  }

  wrap.innerHTML = '';
  const addAllQueue = [];

  for (const partner of uniq) {
    const { node, btn, partnerId } = await renderPartnerCard(partner, end);
    wrap.appendChild(node);
    if (partnerId) addAllQueue.push({ btn, partnerId });
  }

  // "Adicionar todos"
  btnAddAll.disabled = addAllQueue.length === 0;
  btnAddAll.onclick = async () => {
    btnAddAll.classList.add('loading');
    btnAddAll.disabled = true;
    for (const item of addAllQueue) {
      if (item.btn.disabled) continue;
      await sendFriendRequest(item.partnerId, item.btn);
    }
    btnAddAll.classList.remove('loading');
  };
}

/* ================= render ================= */
async function renderPartnerCard(partner, endDate) {
  const tpl = qs('#contactCardTpl');
  const frag = tpl.content.cloneNode(true);
  const root = frag.firstElementChild;

  const nameEl = qsInside(root, '[data-field="name"]');
  const ageEl = qsInside(root, '[data-field="age"]');
  const locEl = qsInside(root, '[data-field="location"]');
  const dateEl = qsInside(root, '[data-field="date"]');
  const timeEl = qsInside(root, '[data-field="time"]');
  const dotEl = qsInside(root, '.dot');
  const addBtn = qsInside(root, '.card__actions .add');

  // normalizações de id
  const partnerId = safeId(partner);

  // nome
  const name = partner.username || partner.name || 'Usuário';
  nameEl.textContent = name;

  // idade (já vem do servidor; se não vier, não exibe)
  const age = Number.isFinite(partner.age) ? Number(partner.age) : null;
  if (age !== null) {
    ageEl.textContent = `${age} anos`;
    ageEl.style.display = '';
  } else {
    ageEl.textContent = '';
    ageEl.style.display = 'none';
  }

  // cidade/UF — exibir só se houver valor
  const city = (partner.city || '').trim();
  const state = (partner.state || '').trim();
  const locTxt = [city, state].filter(Boolean).join(' / ');
  if (locTxt) {
    locEl.textContent = locTxt;
    locEl.style.display = '';
    if (ageEl.style.display === 'none') {
      // se não há idade, remove o ponto separador antes da localização
      dotEl.style.display = 'none';
    } else {
      dotEl.style.display = '';
    }
  } else {
    locEl.textContent = '';
    locEl.style.display = 'none';
    dotEl.style.display = 'none';
  }

  // data/hora (da sessão)
  const when = endDate || parseDate(partner.metAt) || new Date();
  dateEl.textContent = formatDate(when);
  timeEl.textContent = formatTime(when);

  // estado do botão
  if (partnerId) {
    try {
      const status = await friendshipStatus(partnerId);
      applyButtonState(addBtn, status, partnerId, name);
    } catch (e) {
      console.warn('Falha status amizade', e);
      addBtn.textContent = 'Erro';
      addBtn.disabled = true;
      addBtn.classList.add('blocked');
    }
  } else {
    addBtn.textContent = 'Indisponível';
    addBtn.disabled = true;
    addBtn.classList.add('blocked');
  }

  return { node: frag, btn: addBtn, partnerId };
}

/* ================= status & ações ================= */
function applyButtonState(btn, status, partnerId, name) {
  btn.classList.remove('pending', 'friends', 'blocked');
  btn.disabled = false;

  if (status?.blocked) {
    btn.textContent = 'Usuário bloqueado';
    btn.classList.add('blocked'); btn.disabled = true; return;
  }

  switch (status?.friendshipStatus) {
    case 'friends':
    case 'accepted':
      btn.textContent = 'Amigos';
      btn.classList.add('friends'); btn.disabled = true; break;
    case 'pending_sent':
    case 'pending':
      btn.textContent = 'Solicitação enviada';
      btn.classList.add('pending'); btn.disabled = true; break;
    case 'pending_received':
      btn.textContent = 'Responder solicitação';
      btn.classList.add('pending');
      btn.onclick = () => respondToRequest(status.requestId, 'accept');
      break;
    case 'none':
    default:
      btn.textContent = 'Adicionar amigo';
      btn.onclick = () => sendFriendRequest(partnerId, btn);
  }
}

async function friendshipStatus(userId) {
  return fetchJSON(`/friendship/status/${userId}`);
}

async function sendFriendRequest(receiverId, btn) {
  try {
    const old = btn.textContent;
    btn.disabled = true; btn.textContent = 'Enviando...';

    const res = await fetchJSON('/friendship/requests', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      credentials: 'include',
      body: JSON.stringify({ friendId: String(receiverId) })
    });

    if (res?.becameFriends) {
      // ✔ Viraram amigos automaticamente
      btn.textContent = 'Amigos';
      btn.classList.remove('pending');
      btn.classList.add('friends');
      btn.disabled = true;
    } else {
      // ✔ Solicitação enviada (fluxo normal)
      btn.textContent = 'Solicitação enviada';
      btn.classList.add('pending');
      btn.disabled = true;
    }
  } catch (e) {
    console.error(e);
    // Se o backend ainda responder 409 (pendência no mesmo sentido),
    // vale pegar o status para ajustar o botão corretamente.
    try {
      const st = await friendshipStatus(receiverId);
      applyButtonState(btn, st, receiverId);
    } catch {
      btn.textContent = 'Tentar de novo';
      btn.disabled = false;
    }
  }
}

async function respondToRequest(requestId, action){
  try{
    await fetchJSON(`/friendship/requests/${requestId}`, {
      method:'PUT',
      headers:{ 'Content-Type':'application/json' },
      credentials:'include',
      body: JSON.stringify({ action }) // 'accept' | 'reject' | 'cancel'
    });
    window.location.reload();
  }catch(e){
    console.error(e);
    alert('Erro ao responder solicitação.');
  }
}

/* ================= helpers ================= */
function qs(sel, root = document) { return root.querySelector(sel); }
function qsInside(root, sel) { const el = root.querySelector(sel); if (!el) { console.warn('Elemento não encontrado:', sel); } return el; }
function parseDate(v) { try { const d = new Date(v); return isNaN(d) ? null : d; } catch { return null; } }
function pad2(n) { return n < 10 ? '0' + n : '' + n; }
function formatDate(d) { return `${pad2(d.getDate())}/${pad2(d.getMonth() + 1)}/${d.getFullYear()}`; }
function formatTime(d) { return `${pad2(d.getHours())}:${pad2(d.getMinutes())}`; }

function readChatData() {
  try {
    return JSON.parse(sessionStorage.getItem('chatResultsData') || 'null')
      || JSON.parse(localStorage.getItem('chatResultsData') || 'null');
  } catch { return null; }
}

function dedupePartners(list, currentId) {
  const seen = new Set();
  const out = [];
  for (const p of list) {
    const pid = safeId(p);
    if (!pid) continue;
    if (currentId && String(pid) === String(currentId)) continue;
    if (seen.has(pid)) continue;
    seen.add(pid);
    out.push(p);
  }
  return out;
}

function safeId(p) {
  return (p?.userId ?? p?.id ?? p?.user_id ?? p?.sessionId)
    ? String(p.userId ?? p.id ?? p.user_id ?? p.sessionId).replace(/^user_/, '')
    : '';
}

async function fetchJSON(url, opts) {
  const res = await fetch(url, { credentials: 'include', ...(opts || {}) });
  if (!res.ok) {
    let msg = 'Erro de rede';
    try { const j = await res.json(); msg = j?.message || j?.error || msg; } catch { }
    throw new Error(msg);
  }
  return res.json();
}

function emptyState() {
  return `
    <div class="skeleton">
      <p style="margin:0; color:#334155;"><strong>Nenhum parceiro encontrado</strong></p>
      <p style="margin:6px 0 0; color:#64748b;">Volte ao chat e finalize novamente para gerar os resultados.</p>
    </div>
  `;
}
