(() => {
  const $ = (sel, ctx=document) => ctx.querySelector(sel);

  const form = $('#registerForm');
  const alertBox = $('#alert');
  const submitBtn = $('#submitBtn');
  const nameInput = $('#name');
  const genderSelect = $('#gender');
  const emailInput = $('#email');
  const pwInput = $('#password');
  const togglePwBtn = $('#togglePw');
  const pwMeter = $('#pwMeter');

  // Mostrar/ocultar senha
  togglePwBtn.addEventListener('click', () => {
    const isPw = pwInput.type === 'password';
    pwInput.type = isPw ? 'text' : 'password';
  });

  // Medidor simples de força: pontua 0-4
  pwInput.addEventListener('input', () => {
    const v = pwInput.value || '';
    let score = 0;
    if (v.length >= 8) score++;
    if (/[0-9]/.test(v)) score++;
    if (/[a-z]/.test(v)) score++;
    if (/[A-Z!@#$%^&*()\-_=+[\]{};:'",.<>/?`~\\|]/.test(v)) score++;
    pwMeter.setAttribute('data-score', String(Math.min(score,4)));
  });

  function setLoading(isLoading){
    submitBtn.classList.toggle('loading', isLoading);
    submitBtn.disabled = isLoading;
  }

  function showAlert(type, text){
    alertBox.className = 'alert ' + type;
    alertBox.textContent = text;
    alertBox.hidden = !text;
  }

  function clearFieldErrors(){
    ['name','gender','email','password'].forEach(id => {
      const input = document.getElementById(id);
      input?.classList.remove('is-invalid');
    });
    $('#nameError').textContent = '';
    $('#genderError').textContent = '';
    $('#emailError').textContent = '';
    $('#pwError').textContent = '';
  }

  function applyFieldErrors(fieldsMap = {}, errorsList = []){
    // errorsList é útil para exibir hints
    if (fieldsMap.username){
      nameInput.classList.add('is-invalid');
      $('#nameError').textContent = fieldsMap.username;
    }
    if (fieldsMap.gender){
      genderSelect.classList.add('is-invalid');
      $('#genderError').textContent = fieldsMap.gender;
    }
    if (fieldsMap.email){
      emailInput.classList.add('is-invalid');
      $('#emailError').textContent = fieldsMap.email;
    }
    if (fieldsMap.password){
      pwInput.classList.add('is-invalid');
      // tenta achar hint
      const hint = (errorsList.find(e => e.field === 'password') || {}).hint;
      $('#pwError').textContent = fieldsMap.password + (hint ? ` — ${hint}` : '');
    }
  }

  function clientValidate(){
    // validação leve no client (não substitui o backend)
    const errs = {};
    const username = nameInput.value.trim();
    const gender = genderSelect.value;
    const email = emailInput.value.trim();
    const pw = pwInput.value;

    if (username.length < 3) errs.username = 'Usuário deve ter ao menos 3 caracteres.';
    if (!gender) errs.gender = 'Selecione um gênero.';
    if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) errs.email = 'E-mail inválido.';
    if (pw.length < 8 || !/\d/.test(pw)) errs.password = 'Senha fraca. Mínimo 8 caracteres e 1 número.';

    return errs;
  }

  form.addEventListener('submit', async (e) => {
    e.preventDefault();
    clearFieldErrors();
    showAlert('','');

    // 1) validação cliente
    const cErrs = clientValidate();
    if (Object.keys(cErrs).length){
      applyFieldErrors(cErrs, []);
      showAlert('error', 'Corrija os campos destacados e tente novamente.');
      return;
    }

    // 2) prepara payload conforme o backend
    const payload = {
      username: nameInput.value.trim(),
      gender: genderSelect.value,
      email: emailInput.value.trim(),
      password: pwInput.value
    };

    setLoading(true);
    try{
      const resp = await fetch('/auth/register', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify(payload)
      });

      const data = await resp.json().catch(() => ({}));

      if (resp.status === 201 || (resp.ok && data.success)){
        showAlert('success', 'Cadastro realizado com sucesso! Redirecionando…');
        // redirect suave após 1s
        setTimeout(() => window.location.href = '/auth/login', 800);
        return;
      }

      // Erros de validação padronizados (422)
      if (resp.status === 422 && data.code === 'VALIDATION_ERROR'){
        applyFieldErrors(data.fields || {}, data.errors || []);
        showAlert('error', data.summary || 'Dados inválidos.');
        return;
      }

      // Outros erros conhecidos do backend (conflict etc.)
      if (resp.status === 409){
        // Exemplo: e-mail já cadastrado
        applyFieldErrors({ email: data.message || 'E-mail já cadastrado.' });
        showAlert('error', data.message || 'E-mail já cadastrado.');
        return;
      }

      // Fallback genérico
      showAlert('error', data.message || 'Não foi possível completar o cadastro.');
    }catch(err){
      showAlert('error', 'Falha de rede. Verifique sua conexão e tente novamente.');
    }finally{
      setLoading(false);
    }
  });
})();
