// Lobby UX reforçado: avatars, badges, barra de capacidade, countdown visível, toasts.
class LobbyClient {
  constructor() {
    this.socket = null;
    this.state = {
      roomId: null,
      sessionId: null,
      userId: null,
      username: null,
      gender: null
    };

    // refs do modal
    this.modal = null;
    this.modalDialog = null;
    this.modalCloseBtn = null;
    this.modalOkBtn = null;
    this.rulesChipBtn = null;
    this.rulesFullLink = null;
    this._lastFocused = null;

    this.init();
  }

  /* ===================== Boot ===================== */
  init() {
    this.$ = (sel) => document.querySelector(sel);
    this.parseUrlParams();
    this.setupConnection();
    this.setupEventListeners();
    this.setStatus('Aguardando participantes');
    this.updateCapacityBar(0, 2);

    // modal refs
    this.modal = this.$('#rulesModal');
    this.modalDialog = this.$('#rulesModal .modal__dialog');
    this.modalCloseBtn = this.$('#closeRules');
    this.modalOkBtn = this.$('#okRules');
    this.rulesChipBtn = this.$('#rulesChip');
    this.rulesFullLink = this.$('#rulesFullLink');

    this.setupRulesModal();
  }

  parseUrlParams() {
    const urlParams = new URLSearchParams(window.location.search);
    this.state = {
      roomId: urlParams.get('roomId'),
      sessionId: urlParams.get('sessionId'),
      username: urlParams.get('username') || null,
      gender: urlParams.get('gender') || null,
      userId: urlParams.get('userId') || null
    };

    // Mostra um nome legível ao invés do ID cru, se desejar:
    this.$('#roomName').textContent = this.humanizeRoom(this.state.roomId) || (this.state.roomId || '?');
  }

  /* ===================== Socket ===================== */
  setupConnection() {
    this.socket = io({
      withCredentials: true,
      auth: { token: this.getCookie('token') }
    });

    this.socket.on('connect', () => {
      this.joinLobby();
    });

    this.socket.on('transferError', (data) => {
      console.error('Erro na transferência para o chat:', data?.message);
      this.showError('Erro ao entrar no chat. Tente novamente.');
      this.toast('Erro ao entrar no chat.');
      setTimeout(() => window.location.href = '/', 3000);
    });

    this.socket.on('lobbyUpdate', (data) => this.updateLobby(data));

    this.socket.on('countdownUpdate', (data) => this.showCountdown(data.countdown));

    // ⚠️ NÃO afirmar que alguém saiu aqui; countdown pode ser cancelado por outros motivos
    this.socket.on('countdownCancelled', () => {
      this.setStatus('Aguardando participantes');
      this.$('#countdownWrap').hidden = true;
      // limpar qualquer erro antigo para não sugerir saída de usuário
      const err = this.$('#errorMessage');
      if (err) err.hidden = true;
      this.toast('Pareamento adiado — aguardando participantes');
    });

    // Novo: notificação explícita quando alguém REALMENTE sai do lobby
    this.socket.on('lobbyUserLeft', (data = {}) => {
      const name = data.username || 'Um participante';
      this.showError(`${name} saiu. Aguardando substituição…`);
      this.toast(`${name} saiu do lobby`);
      this.setStatus('Aguardando substituição…');
      setTimeout(() => { const e = this.$('#errorMessage'); if (e) e.hidden = true; }, 3000);
    });

    // proteção defensiva (raro no lobby)
    this.socket.on('partnerDisconnected', (data) => {
      this.toast(`${data?.username || 'Participante'} desconectou. Aguardando reconexão...`);
      this.setStatus('Aguardando reconexão…');
    });

    this.socket.on('redirectToChat', (data) => {
      if (!data?.data?.partner?.socketId) {
        console.error('Dados do parceiro inválidos:', data);
        this.showError('Erro: Parceiro não disponível');
        this.toast('Erro na alocação do parceiro');
        return;
      }
      sessionStorage.setItem('chatTransferData', JSON.stringify(data));
      window.location.href = `/chat?roomId=${data.data.roomId}&sessionId=${data.data.currentUser.sessionId}`;
    });

    this.socket.on('disconnect', () => {
      this.toast('Conexão perdida. Tentando reconectar...');
      this.setStatus('Reconectando…');
    });
  }

  /* ===================== Events ===================== */
  setupEventListeners() {
    this.$('#leaveLobby')?.addEventListener('click', () => {
      this.socket.emit('leaveLobby', {
        roomId: this.state.roomId,
        sessionId: this.state.sessionId
      });
      window.location.href = '/';
    });
  }

  /* ===================== Helpers ===================== */
  getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
    return null;
  }

  humanizeRoom(roomId) {
    if (!roomId) return null;
    const mapCat = {
      heterossexual: 'Heterossexual',
      homossexual_masculino: 'Homossexual Masculino',
      homossexual_feminino: 'Homossexual Feminino',
      lgbt: 'LGBT+'
    };
    const m = roomId.match(/^(heterossexual|homossexual_masculino|homossexual_feminino|lgbt)_sala(\d)$/i);
    if (!m) return roomId;
    const [, kind, sala] = m;
    return `${mapCat[kind] || kind} · Sala ${sala}`;
  }

  avatarFromName(name) {
    const n = (name || 'U U').trim().split(/\s+/);
    const letters = (n[0]?.[0] || 'U') + (n[1]?.[0] || '');
    return letters.toUpperCase().slice(0, 2);
  }

  genderBadgeClass(gender) {
    if (!gender) return '';
    const g = String(gender).toLowerCase();
    if (g === 'male') return 'badge--male';
    if (g === 'female') return 'badge--female';
    return '';
  }

  setStatus(text) {
    const el = this.$('#statusLabel');
    if (el) el.textContent = text;
  }

  updateCapacityBar(current, max) {
    const pct = Math.max(0, Math.min(100, Math.round((current / Math.max(1, max)) * 100)));
    const bar = this.$('#capacityBar');
    const prog = this.$('.progress');
    if (bar) bar.style.width = `${pct}%`;
    if (prog) {
      prog.setAttribute('aria-valuenow', String(pct));
      prog.title = `Ocupação ${pct}%`;
    }
  }

  toast(message) {
    const el = this.$('#toast');
    if (!el) return;
    el.textContent = message;
    el.hidden = false;
    clearTimeout(this._toastTimer);
    this._toastTimer = setTimeout(() => (el.hidden = true), 2500);
  }

  showError(message) {
    const errorEl = this.$('#errorMessage');
    if (!errorEl) return;
    errorEl.textContent = message;
    errorEl.hidden = false;
  }

  showCountdown(seconds) {
    const wrap = this.$('#countdownWrap');
    const num = this.$('#countdownNumber');
    if (!wrap || !num) return;
    this.setStatus('Sala completa! Preparando pareamento…');
    wrap.hidden = false;
    num.textContent = String(seconds);
  }

  /* ===================== Business ===================== */
  joinLobby() {
    const joinData = {
      roomId: this.state.roomId,
      sessionId: this.state.sessionId,
      socketId: this.socket.id,
      userId: this.state.userId || undefined,
      // username/gender podem vir nulos — o servidor busca do token/DB
      username: this.state.username || undefined,
      gender: this.state.gender || undefined
    };

    if (!joinData.roomId || !joinData.sessionId) {
      this.showError('Dados incompletos para entrar na sala');
      setTimeout(() => window.location.href = '/', 1500);
      return;
    }

    this.socket.emit('joinLobby', joinData, (response) => {
      if (response && response.status === 'error') {
        this.showError(response.message);
        this.toast('Não foi possível entrar na sala.');
        setTimeout(() => window.location.href = '/', 1500);
      }
    });
  }

  updateLobby(data) {
    if (!data) return;

    const current = Number(data.currentUsers || 0);
    const max = Number(data.maxUsers || 2);

    this.$('#userCount').textContent = current;
    this.$('#maxUsers').textContent = max;
    this.updateCapacityBar(current, max);

    // atualizar link "ver página completa" dentro do modal, se existir rulesUrl
    if (this.rulesFullLink) {
      if (data.rulesUrl) {
        this.rulesFullLink.href = data.rulesUrl;
        this.rulesFullLink.hidden = false;
      } else {
        this.rulesFullLink.hidden = true;
      }
    }

    const usersList = this.$('#usersList');
    usersList.innerHTML = '';

    if (data.userData) {
      Object.values(data.userData).forEach(user => {
        const isSelf = user.sessionId === this.state.sessionId;

        const item = document.createElement('div');
        item.className = 'user';

        const avatar = document.createElement('div');
        avatar.className = 'user__avatar';
        avatar.textContent = this.avatarFromName(user.username);

        const main = document.createElement('div');
        main.className = 'user__main';

        const name = document.createElement('div');
        name.className = 'user__name';
        name.textContent = user.username || 'Usuário';

        const meta = document.createElement('div');
        meta.className = 'user__meta';

        if (isSelf) {
          const selfB = document.createElement('span');
          selfB.className = 'badge badge--self';
          selfB.textContent = 'Você';
          meta.appendChild(selfB);
        }

        main.appendChild(name);
        main.appendChild(meta);

        item.appendChild(avatar);
        item.appendChild(main);

        usersList.appendChild(item);
      });
    }

    // Estado textual
    if (current < max) {
      this.setStatus('Aguardando participantes');
      this.$('#countdownWrap').hidden = true;
    } else {
      this.setStatus('Sala completa! Preparando pareamento…');
    }
  }

  /* ===================== Modal (Regras) ===================== */
  setupRulesModal() {
    if (!this.modal || !this.rulesChipBtn) return;

    // abrir
    this.rulesChipBtn.addEventListener('click', (e) => {
      e.preventDefault();
      this.openModal();
    });

    // fechar
    this.modalCloseBtn?.addEventListener('click', () => this.closeModal());
    this.modalOkBtn?.addEventListener('click', () => this.closeModal());
    this.modal.addEventListener('click', (e) => {
      if (e.target && e.target.hasAttribute('data-close-modal')) this.closeModal();
    });

    // ESC fecha
    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape' && this.modal.classList.contains('is-open')) {
        this.closeModal();
      }
    });

    // foco preso
    this.modal.addEventListener('keydown', (e) => {
      if (e.key !== 'Tab') return;
      const focusables = this.getFocusable(this.modal);
      if (!focusables.length) return;
      const first = focusables[0];
      const last = focusables[focusables.length - 1];
      if (e.shiftKey && document.activeElement === first) {
        e.preventDefault(); last.focus();
      } else if (!e.shiftKey && document.activeElement === last) {
        e.preventDefault(); first.focus();
      }
    });
  }

  openModal() {
    this._lastFocused = document.activeElement;
    this.modal.classList.add('is-open');
    this.modal.setAttribute('aria-hidden', 'false');
    document.body.classList.add('no-scroll');

    // foca no título ou no botão fechar
    setTimeout(() => {
      (this.modalDialog || this.modalCloseBtn || this.modal).focus();
    }, 0);
  }

  closeModal() {
    this.modal.classList.remove('is-open');
    this.modal.setAttribute('aria-hidden', 'true');
    document.body.classList.remove('no-scroll');
    if (this._lastFocused && typeof this._lastFocused.focus === 'function') {
      this._lastFocused.focus();
    }
  }

  getFocusable(root) {
    return Array.from(
      root.querySelectorAll(
        'a[href], button, textarea, input, select, [tabindex]:not([tabindex="-1"])'
      )
    ).filter(el => !el.hasAttribute('disabled') && !el.getAttribute('aria-hidden'));
  }
}

document.addEventListener('DOMContentLoaded', () => new LobbyClient());
