// Helper 401 -> refresh
async function fetchWithRefresh(input, init = {}) {
  const exec = () => fetch(input, { credentials: 'include', ...init });
  let res = await exec();
  if (res.status === 401) {
    const r = await fetch('/auth/refresh', { method: 'POST', credentials: 'include' });
    if (r.ok) res = await exec();
  }
  return res;
}

// Verifica se resposta é JSON, senão lança erro com trecho de HTML
async function parseJsonSafe(res) {
  const ct = res.headers.get('content-type') || '';
  if (!ct.includes('application/json')) {
    const text = await res.text();
    throw new Error(`Resposta não-JSON (${res.status}). Trecho: ${text.slice(0, 180)}`);
  }
  return res.json();
}

function calcAge(iso) {
  const d = new Date(iso);
  if (isNaN(d)) return null;
  const now = new Date();
  let age = now.getFullYear() - d.getFullYear();
  const m = now.getMonth() - d.getMonth();
  if (m < 0 || (m === 0 && now.getDate() < d.getDate())) age--;
  return age;
}

// Catálogo para mapear códigos -> labels (idêntico ao profile.js)
const LABELS = {
  relationship_goals: {
    amizade: 'Amizade', casual: 'Casual', namoro: 'Namoro', casamento: 'Casamento',
    networking: 'Networking', parceria: 'Parceria criativa', nao_sei: 'Não sei ainda'
  },
  relationship_type: {
    monogamico: 'Monogâmico', aberto: 'Aberto', poliamor: 'Poliamor', discreto: 'Discreto', nao_sei: 'Não sei ainda'
  },
  has_children: {
    tenho_nao_quero_mais: 'Tenho, e não quero mais um.', tenho: 'Tenho',
    tenho_quero_mais: 'Tenho e quero mais.', nao_tenho_nao_quero: 'Não tenho, e não quero',
    nao_tenho_aberto: 'Não tenho mas estou aberto(a)', prefiro_nao_dizer: 'Prefiro não dizer'
  },
  drinks: { nao: 'Não', socialmente: 'Socialmente', raramente: 'Raramente', frequentemente: 'Frequentemente' },
  smokes: { nao: 'Não', ocasional: 'Ocasional', vape: 'Vape', sim: 'Sim' },
  political_orientation: {
    esquerda: 'Esquerda', centro_esquerda: 'Centro-esquerda', centro: 'Centro',
    centro_direita: 'Centro-direita', direita: 'Direita', libertario: 'Libertário',
    independente: 'Independente', prefiro_nao_dizer: 'Prefiro não dizer'
  },
  religion: {
    catolico: 'Católico(a)', evangelico: 'Evangélico(a)', espirita: 'Espírita',
    umbanda_candomble: 'Umbanda/Candomblé', judaico: 'Judaico(a)', muculmano: 'Muçulmano(a)',
    budista: 'Budista', agnostico: 'Agnóstico(a)', ateu: 'Ateu/Atéia', outra: 'Outra', prefiro_nao_dizer: 'Prefiro não dizer'
  },
  kitchen_persona: {
    chef_fim_de_semana: 'Chef de fim de semana', mestre_microondas: 'Mestre do micro-ondas',
    rei_da_airfryer: 'Rei/Rainha da airfryer', tempero_da_vovo: 'Tempero da vovó', queima_ate_gelo: 'Queima até gelo'
  },
  diet_style: {
    conto_macros: 'Conto macros', low_carb: 'Low-carb quando dá', vegetariano: 'Vegetariano(a)',
    vegano: 'Vegano(a)', onivoro_feliz: 'Onívoro(a) feliz', segunda_sem_carne: 'Só segunda sem carne',
    jejum_intermitente: 'Jejum intermitente', nao_faco_ideia: 'Não faço ideia 😅'
  },
  pets: {
    team_dogs: 'Team dogs', team_cats: 'Team cats', amo_todos: 'Amo todos',
    alergia_mas_tento: 'Tenho alergia mas tento', peixes_e_plantas: 'Peixes e plantas', prefiro_plantas: 'Prefiro plantas'
  },
  coffee: {
    sempre: 'Sempre!', com_acucar: 'Só com açúcar', com_leite: 'Com leite',
    descafeinado: 'Descafeinado', cha: 'Chá, por favor', nao_curto: 'Não curto'
  },
  sports_role: {
    atleta_dedicado: 'Atleta dedicado(a)', torcedor_de_sofa: 'Torcedor(a) de sofá',
    domingo_no_parque: 'Domingo no parque', marombeiro: 'Marombeiro(a)',
    yoga_pilates: 'Yoga/Pilates', corrida: 'Corrida', ciclista: 'Ciclista'
  },
  party_style: {
    dj_improvisado: 'DJ improvisado', guardiao_da_bolsa: 'Guardião(ã) da bolsa',
    dono_da_roda: 'Dono(a) da roda de conversa', primeiro_na_pista: 'Primeiro(a) na pista',
    volta_com_batata: 'Some e volta com batata', fotografo: 'Fotógrafo(a) do rolê',
    deus_me_livre: 'Deus me livre', prefiro_nem_ir: 'Prefiro nem ir'
  },
  gangster_persona: {
    estrategista: 'O estrategista', motorista_fuga: 'O motorista de fuga',
    hacker: 'O hacker', pacificador: 'O pacificador', esquece_senha: 'O que esquece a senha'
  }
};

// Interesse: mapa slug->label (carregado do backend)
let INTEREST_CATALOG = []; // [{slug,label,...}]

document.addEventListener('DOMContentLoaded', async () => {
  const profileId = window.location.pathname.split('/').pop();
  const actionButtons = document.getElementById('actionButtons');
  const galleryGrid = document.getElementById('galleryGrid');

  try {
    // Carrega catálogo de interesses (para resolver slugs)
    try {
      const r = await fetchWithRefresh('/auth/interests');
      if (r.ok) {
        const data = await parseJsonSafe(r);
        INTEREST_CATALOG = Array.isArray(data.items) ? data.items : [];
      }
    } catch { /* silencioso */ }

    // 1) Perfil (explorer/view respeita bloqueio/visibilidade)
    const profileResponse = await fetchWithRefresh(`/friendship/explorer/view/${profileId}`);
    if (!profileResponse.ok) {
      if (profileResponse.status === 403) { renderBlockedState(); return; }
      throw new Error(await profileResponse.text() || 'Erro ao carregar perfil');
    }
    const profileData = await parseJsonSafe(profileResponse);
    renderProfile(profileData);

    // 2) Status amizade/bloqueio
    const statusResponse = await fetchWithRefresh(`/friendship/status/${profileId}`);
    if (!statusResponse.ok) throw new Error(await statusResponse.text() || 'Erro ao verificar status');
    const statusData = await parseJsonSafe(statusResponse);
    renderActionButtons(statusData, profileId);

    // 3) Galeria
    await loadGalleryImages(profileId);
  } catch (e) {
    console.error(e);
    actionButtons.innerHTML = `
      <section class="card"><div class="card-body">
        <div class="gallery-empty">
          <i class="fas fa-exclamation-triangle"></i> ${e.message}
        </div>
      </div></section>`;
  }

  async function loadGalleryImages(userId) {
    try {
      galleryGrid.innerHTML = `
        <div class="gallery-empty"><i class="fas fa-spinner fa-spin"></i> Carregando galeria...</div>`;
      const res = await fetchWithRefresh(`/friendship/galeria/${userId}`);
      if (!res.ok) {
        if (res.status === 403) {
          galleryGrid.innerHTML = `
            <div class="gallery-empty"><i class="fas fa-ban"></i> Você não pode ver a galeria deste usuário.</div>`;
          return;
        }
        throw new Error(await res.text() || 'Erro ao carregar galeria');
      }
      const data = await parseJsonSafe(res);
      renderGallery(Array.isArray(data.images) ? data.images : []);
    } catch (err) {
      console.error(err);
      galleryGrid.innerHTML = `
        <div class="gallery-empty"><i class="fas fa-exclamation-circle"></i> ${err.message}</div>`;
    }
  }
});

function labelify(group, value) {
  if (!value) return '—';
  const map = LABELS[group] || {};
  return map[value] || value;
}

function renderProfile(u) {
  // Header
  document.getElementById('username').textContent = u.username || 'Não informado';
  document.getElementById('userGender').textContent =
    u.gender === 'male' ? 'Masculino' : u.gender === 'female' ? 'Feminino' : 'Não informado';

  if (u.profile_picture) document.getElementById('profilePicture').src = u.profile_picture;

  const pron = (u.pronouns || '').trim();
  const occ = (u.occupation || '').trim();
  const loc = [u.city, u.state].filter(Boolean).join(' / ');

  const pronEl = document.getElementById('pronounsText');
  const occEl = document.getElementById('occupationText');
  const locEl = document.getElementById('locationText');

  if (pron) { pronEl.textContent = pron; pronEl.style.display = 'inline-flex'; }
  if (occ) { occEl.textContent = occ; occEl.style.display = 'inline-flex'; }
  if (loc) { locEl.textContent = loc; locEl.style.display = 'inline-flex'; }

  // Bio
  document.getElementById('bioText').textContent = u.bio || '—';

  // Básico
  // Idade (sem exibir a data de nascimento)
  const birthSpan = document.getElementById('birthDate');
  const age = u.birth_date ? calcAge(u.birth_date) : null;
  if (age != null) {
    birthSpan.textContent = `${age} anos`;
    const label = birthSpan?.previousElementSibling;
    if (label && label.tagName === 'LABEL') label.textContent = 'Idade';
  } else {
    birthSpan.textContent = '—';
  }
  document.getElementById('height').textContent = u.height ? `${u.height} cm` : '—';
  document.getElementById('weight').textContent = u.weight ? `${u.weight} kg` : '—';

  // Relacionamento
  document.getElementById('relationship_goals').textContent = labelify('relationship_goals', u.relationship_goals);
  document.getElementById('relationship_type').textContent = labelify('relationship_type', u.relationship_type);
  document.getElementById('has_children').textContent = labelify('has_children', u.has_children);
  document.getElementById('drinks').textContent = labelify('drinks', u.drinks);
  document.getElementById('smokes').textContent = labelify('smokes', u.smokes);
  document.getElementById('political_orientation').textContent = labelify('political_orientation', u.political_orientation);
  document.getElementById('religion').textContent = labelify('religion', u.religion);

  // Preferências
  document.getElementById('kitchen_persona').textContent = labelify('kitchen_persona', u.kitchen_persona);
  document.getElementById('diet_style').textContent = labelify('diet_style', u.diet_style);
  document.getElementById('pets').textContent = labelify('pets', u.pets);
  document.getElementById('coffee').textContent = labelify('coffee', u.coffee);
  document.getElementById('sports_role').textContent = labelify('sports_role', u.sports_role);
  document.getElementById('party_style').textContent = labelify('party_style', u.party_style);
  document.getElementById('gangster_persona').textContent = labelify('gangster_persona', u.gangster_persona);

  // Interesses (slugs -> labels se catálogo disponível)
  const interests = Array.isArray(u.interests) ? u.interests : safeParseArray(u.interests);
  renderTagList('interestsTags', resolveInterestLabels(interests));

  // Com você eu quero / Dia perfeito
  const withYou = Array.isArray(u.with_you) ? u.with_you : safeParseArray(u.with_you);
  const perfectDay = Array.isArray(u.perfect_day) ? u.perfect_day : safeParseArray(u.perfect_day);
  renderTagList('withYouTags', withYou);
  renderTagList('perfectDayTags', perfectDay);

  // Redes sociais
  const links = Array.isArray(u.social_links) ? u.social_links : safeParseArray(u.social_links);
  renderSocialList(links);
}

function safeParseArray(val) {
  if (!val) return [];
  try { const v = typeof val === 'string' ? JSON.parse(val) : val; return Array.isArray(v) ? v : []; }
  catch { return []; }
}

function resolveInterestLabels(slugs) {
  if (!INTEREST_CATALOG?.length) return slugs;
  const map = new Map(INTEREST_CATALOG.map(i => [i.slug, i.label]));
  return slugs.map(s => map.get(s) || s);
}

function renderTagList(id, items) {
  const el = document.getElementById(id);
  el.innerHTML = '';
  if (!items?.length) {
    el.innerHTML = `<span class="muted">—</span>`;
    return;
  }
  items.forEach(txt => {
    const b = document.createElement('button');
    b.type = 'button';
    b.className = 'tag is-selected';
    b.textContent = txt;
    b.disabled = true;
    el.appendChild(b);
  });
}

function renderSocialList(items) {
  const list = document.getElementById('socialLinksList');
  list.innerHTML = '';
  if (!items?.length) {
    list.innerHTML = `<div class="muted">—</div>`;
    return;
  }
  items.forEach(x => {
    const it = document.createElement('div');
    it.className = 'social-item';
    it.innerHTML = `
      <i class="fa-solid fa-link"></i>
      <strong>${escapeHtml(x.network || 'Link')}</strong>
      <a href="${escapeAttr(x.url || '#')}" target="_blank" rel="noopener">${x.url || ''}</a>
    `;
    list.appendChild(it);
  });
}
function escapeHtml(s = '') { return String(s).replace(/[&<>"']/g, m => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[m])); }
function escapeAttr(s = '') { return String(s).replace(/"/g, '&quot;'); }

function renderGallery(images) {
  const galleryGrid = document.getElementById('galleryGrid');
  galleryGrid.innerHTML = '';
  if (!images.length) {
    galleryGrid.innerHTML = `<div class="gallery-empty"><i class="fas fa-images"></i> Nenhuma imagem na galeria</div>`;
    return;
  }
  images.forEach(img => {
    const item = document.createElement('div');
    item.className = 'gallery-item';
    item.innerHTML = `<img src="${img.image_path}" alt="Imagem" loading="lazy"/>`;
    galleryGrid.appendChild(item);
  });
}

function renderBlockedState() {
  const wrap = document.querySelector('.container');
  wrap.innerHTML = `
    <section class="card">
      <div class="card-body" style="text-align:center">
        <i class="fas fa-ban" style="font-size:42px;color:var(--danger)"></i>
        <h2 class="h2" style="margin:10px 0 6px">Você não pode visualizar este perfil</h2>
        <p class="muted">O usuário está bloqueado ou você foi bloqueado.</p>
        <a class="btn btn-outline back-btn" style="margin-top:12px" href="/dashboard/rooms">
          <i class="fas fa-arrow-left"></i> Voltar
        </a>
      </div>
    </section>`;
}

function renderActionButtons(status, profileId) {
  const wrap = document.getElementById('actionButtons');
  wrap.innerHTML = '';

  // Bloqueio sempre vence
  if (status.blocked) {
    wrap.innerHTML = `
      <section class="card"><div class="card-body">
        <div class="gallery-empty"><i class="fas fa-ban"></i> Você bloqueou este usuário ou foi bloqueado.</div>
      </div></section>`;
    return;
  }

  const addBtn = (cls, html, onClick, disabled = false) => {
    const btn = document.createElement('button');
    btn.className = `btn ${cls}`.trim();
    btn.innerHTML = html;
    btn.disabled = disabled;
    if (onClick) btn.onclick = onClick;
    wrap.appendChild(btn);
    return btn;
  };

  switch (status.friendshipStatus) {
    case 'friends':
      addBtn('btn-primary', `<i class="fas fa-envelope"></i> Enviar mensagem`,
        () => window.location.href = `/chatFriends/room?id_friend=${profileId}`);
      addBtn('btn-ghost', `<i class="fas fa-user-minus"></i> Desfazer amizade`,
        () => removeFriend(profileId));
      break;

    case 'pending_sent':
      addBtn('btn-ghost', `<i class="fas fa-clock"></i> Solicitação enviada`, null, true);
      addBtn('btn-danger-soft', `<i class="fas fa-times"></i> Cancelar solicitação`,
        () => respondToRequest(status.requestId, 'cancel'));
      break;

    case 'pending_received':
      addBtn('btn-primary', `<i class="fas fa-check"></i> Aceitar amizade`,
        () => respondToRequest(status.requestId, 'accept'));
      addBtn('btn-danger-soft', `<i class="fas fa-times"></i> Rejeitar`,
        () => respondToRequest(status.requestId, 'reject'));
      break;

    case 'rejected':
    case 'cancelled':
    case 'none':
    default:
      addBtn('btn-primary', `<i class="fas fa-user-plus"></i> Solicitar amizade`,
        sendFriendRequest);
      break;
  }

  addBtn('btn-outline', `<i class="fas fa-ban"></i> Bloquear usuário`, () => handleBlockAction(profileId));
}

// ==== Ações (amizade/bloqueio) ====
async function sendFriendRequest() {
  const btn = this; const original = btn.innerHTML;
  try {
    btn.classList.add('loading'); btn.disabled = true; btn.innerHTML = `<span class="spinner"></span><span class="btn-text"> Enviando…</span>`;
    const profileId = window.location.pathname.split('/').pop();
    const res = await fetchWithRefresh('/friendship/requests', {
      method: 'POST', headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ friendId: profileId }),
    });
    if (!res.ok) throw new Error(await res.text());
    alert('Solicitação enviada com sucesso!');
    location.reload();
  } catch (e) {
    alert(e.message); btn.disabled = false; btn.innerHTML = original; btn.classList.remove('loading');
  }
}

async function respondToRequest(requestId, action) {
  const buttons = document.querySelectorAll('#actionButtons .btn');
  const older = Array.from(buttons).map(b => b.innerHTML);
  try {
    buttons.forEach(b => { b.disabled = true; b.innerHTML = `<span class="spinner"></span><span class="btn-text"> Processando…</span>`; b.classList.add('loading'); });
    const res = await fetchWithRefresh(`/friendship/requests/${requestId}`, {
      method: 'PUT', headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ action }) // 'accept' | 'reject' | 'cancel'
    });
    if (!res.ok) throw new Error(await res.text());
    alert('Atualizado!'); location.reload();
  } catch (e) {
    alert(e.message);
    buttons.forEach((b, i) => { b.disabled = false; b.innerHTML = older[i]; b.classList.remove('loading'); });
  }
}

async function removeFriend(friendId) {
  if (!confirm('Tem certeza que deseja desfazer a amizade?')) return;
  try {
    const res = await fetchWithRefresh(`/friendship/friends/${friendId}`, { method: 'DELETE' });
    if (!res.ok) throw new Error(await res.text());
    alert('Amizade removida'); location.reload();
  } catch (e) { alert(e.message); }
}

async function handleBlockAction(targetId) {
  if (!confirm('Bloquear este usuário? Vocês deixarão de ser amigos e não poderão interagir.')) return;
  try {
    const res = await fetchWithRefresh('/friendship/blocks', {
      method: 'POST', headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ targetId })
    });
    if (!res.ok) throw new Error(await res.text());
    alert('Usuário bloqueado.'); location.href = '/dashboard/rooms';
  } catch (e) { alert(e.message); }
}
