document.addEventListener('DOMContentLoaded', initHistory);

async function initHistory() {
  const container = qs('#historyContainer');

  try {
    // userId vem do path: /chatHistory/<id>
    const parts = window.location.pathname.split('/').filter(Boolean);
    const userId = parts[parts.length - 1];

    const res = await fetch(`/chatHistory/history/${userId}`, { credentials: 'include' });
    if (!res.ok) {
      const { error } = await safeJson(res);
      throw new Error(error || 'Erro ao carregar histórico');
    }
    const data = await res.json();

    if (!data.history || data.history.length === 0) {
      container.innerHTML = emptyState();
      return;
    }

    container.innerHTML = '';
    data.history.forEach(renderCard(container));
  } catch (err) {
    console.error('Erro histórico:', err);
    qs('#historyContainer').innerHTML = errorState(err.message);
  }
}

/* ========= render ========= */
function renderCard(container) {
  return (chat) => {
    const end = parseDate(chat.chat_end_time);
    const dateStr = end ? end.toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit', year: 'numeric' }) : '--/--/----';
    const timeStr = end ? end.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' }) : '--:--';
    const partnerAge = Number.isFinite(chat.partner_age) ? `${chat.partner_age} anos` : 'idade não informada';

    const tpl = qs('#historyCardTpl');
    const node = tpl.content.cloneNode(true);

    const link = node.querySelector('.card-link');
    link.href = `/explorer/ver/${chat.partner_id}`;

    node.querySelector('.who__name').textContent = chat.partner_username || 'Parceiro';
    node.querySelector('.badge--age').textContent = partnerAge;
    node.querySelector('.when__date').textContent = dateStr;
    node.querySelector('.when__time').textContent = timeStr;

    container.appendChild(node);
  };
}

/* ========= helpers ========= */
function parseDate(v) {
  try {
    const d = new Date(v);
    return isNaN(d) ? null : d;
  } catch { return null; }
}
function qs(sel, root = document) { return root.querySelector(sel); }
async function safeJson(res) {
  try { return await res.json(); } catch { return {}; }
}

/* ========= states ========= */
function emptyState() {
  return `
    <div class="empty">
      <h3>Nenhuma conversa encontrada</h3>
      <p>Você ainda não possui conversas em seu histórico.</p>
      <a class="btn btn-outline" href="/dashboard/rooms">Voltar para o início</a>
    </div>
  `;
}
function errorState(message) {
  return `
    <div class="empty empty--error">
      <h3>Erro ao carregar histórico</h3>
      <p>${escapeHtml(message)}</p>
      <button class="btn btn-outline" onclick="window.location.reload()">Tentar novamente</button>
    </div>
  `;
}
function escapeHtml(s='') {
  return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m]));
}
