document.addEventListener('DOMContentLoaded', async () => {
  const urlParams = new URLSearchParams(window.location.search);
  const friendId = urlParams.get('id_friend');
  if (!friendId) { alert('Amigo não especificado.'); return; }

  // Helper: cookie
  function getCookie(name) {
    const m = document.cookie.match(new RegExp('(?:^|; )' + name.replace(/([.*+?^${}()|[\\]\\\\])/g, '\\\\$1') + '=([^;]*)'));
    return m ? decodeURIComponent(m[1]) : undefined;
  }

  // Socket
  const socket = io({ auth: { token: getCookie('token') } });

  // Carrega user + amigo + meta de amizade
  const [userRes, friendRes, metaRes] = await Promise.all([
    fetch('/auth/me', { credentials: 'include' }),
    fetch(`/chatFriends/friend/${friendId}`, { credentials: 'include' }),
    fetch(`/chatFriends/friendship/${friendId}`, { credentials: 'include' })
  ]);
  if (!userRes.ok || !friendRes.ok) { alert('Erro ao carregar dados'); return; }

  const user = await userRes.json();
  const friend = await friendRes.json();
  const meta = metaRes.ok ? await metaRes.json().catch(() => ({})) : {};

  // Header: avatar, nome, idade, amigos desde
  const friendNameEl = document.getElementById('friendName');
  const friendAvatarEl = document.getElementById('friendAvatar');
  const friendAgeEl = document.getElementById('friendAge');
  const friendSinceEl = document.getElementById('friendSince');

  friendNameEl.textContent = friend.username || 'Amigo';
  friendAvatarEl.src = friend.profile_picture || '/img/user_placeholder.png';

  if (friend.birth_date) {
    friendAgeEl.textContent = `Idade: ${calcAge(friend.birth_date)} anos`;
  }
  if (meta?.friendship_date) {
    friendSinceEl.textContent = `Amigos desde: ${formatDate(meta.friendship_date)}`;
  }

  const currentUserId = user.id;

  // Marca lidas ao abrir
  try { await fetch(`/chatFriends/mark-read/${friendId}`, { method: 'POST', credentials: 'include' }); } catch {}

  // ============== CHAT RENDER ==============
  const chatList = document.getElementById('chatContainer');
  chatList.innerHTML = '';

  function dayKey(d) { const dt = new Date(d); return dt.getFullYear() + '-' + (dt.getMonth() + 1) + '-' + dt.getDate(); }
  let lastDay = null;

  function appendDaySeparator(dateStr) {
    const li = document.createElement('li');
    li.className = 'day-sep';
    li.textContent = formatDate(dateStr);
    chatList.appendChild(li);
  }

  function addMessageToChat(msg, isSent) {
    const ts = msg.sent_at || msg.created_at || new Date().toISOString();
    const dk = dayKey(ts);
    if (dk !== lastDay) { appendDaySeparator(ts); lastDay = dk; }

    const li = document.createElement('li');
    li.className = `msg ${isSent ? 'msg--out' : 'msg--in'}`;
    li.dataset.mid = msg.id ?? '';

    li.innerHTML = `
      <div class="msg__text">${escapeHtml(String(msg.message || ''))}</div>
      <div class="msg__meta">
        <span class="msg__time">${formatTime(ts)}</span>
        ${isSent ? `
          <span class="msg__status ${Number(msg.is_read) ? 'read' : ''}">
            <i class="fa-solid fa-check"></i>
            <i class="fa-solid fa-check-double"></i>
          </span>` : ''}
      </div>
    `;
    chatList.appendChild(li);
    chatList.scrollTop = chatList.scrollHeight;
  }

  // Carrega histórico
  await (async function loadMessages() {
    const res = await fetch(`/chatFriends/messages/${friendId}`, { credentials: 'include' });
    if (!res.ok) return;
    let messages = [];
    try { messages = await res.json(); if (!Array.isArray(messages)) messages = messages ? [messages] : []; } catch {}
    messages.forEach(m => addMessageToChat(m, String(m.sender_id) === String(currentUserId)));
  })();

  // Recebe novas mensagens
  socket.on('new_message', async (message) => {
    const mine = String(message.sender_id) === String(currentUserId);
    const peer = String(message.sender_id) === String(friendId) && String(message.receiver_id) === String(currentUserId);
    const toPeer = mine && String(message.receiver_id) === String(friendId);

    if (peer || toPeer) {
      addMessageToChat(message, mine);
      // se recebi do amigo, marcar como lida
      if (peer) {
        try { await fetch(`/chatFriends/mark-read/${friendId}`, { method: 'POST', credentials: 'include' }); } catch {}
      }
    }
  });

  // Alguém marcou como lidas → se foi o amigo, pintar ✓✓ nas minhas mensagens
  socket.on('messages_read', ({ by }) => {
    if (String(by) !== String(friendId)) return; // lido pelo amigo desta conversa
    document.querySelectorAll('.msg.msg--out .msg__status').forEach(s => s.classList.add('read'));
  });

  // Enviar
  const form = document.getElementById('messageForm');
  const sendBtn = document.getElementById('sendButton');

  form.addEventListener('submit', async (e) => {
    e.preventDefault();
    const input = document.getElementById('messageInput');
    const text = input.value.trim();
    if (!text) return;

    // opcional: feedback de loading no botão (usa spinner do global.css)
    // sendBtn.classList.add('loading');

    const res = await fetch('/chatFriends/send', {
      method: 'POST',
      credentials: 'include',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ receiverId: friendId, message: text })
    });

    // opcional: remover loading
    // sendBtn.classList.remove('loading');

    if (res.ok) {
      input.value = '';
      // 'new_message' chegará pelo socket com sent_at
    } else {
      alert('Erro ao enviar mensagem.');
    }
  });

  // ===== helpers =====
  function escapeHtml(s) { return String(s).replace(/[&<>\"'`=\/]/g, c => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;', '/': '&#x2F;', '`': '&#x60;', '=': '&#x3D;' }[c])); }
  function pad(n) { return n < 10 ? '0' + n : '' + n; }
  function formatDate(d) {
    const dt = new Date(d);
    if (isNaN(dt)) return '--/--/----';
    return pad(dt.getDate()) + '/' + pad(dt.getMonth() + 1) + '/' + dt.getFullYear();
  }
  function formatTime(d) {
    const dt = new Date(d);
    if (isNaN(dt)) return '--:--';
    return pad(dt.getHours()) + ':' + pad(dt.getMinutes());
  }
  function calcAge(dateStr) {
    const b = new Date(dateStr); if (isNaN(b)) return '--';
    const t = new Date();
    let age = t.getFullYear() - b.getFullYear();
    const m = t.getMonth() - b.getMonth();
    if (m < 0 || (m === 0 && t.getDate() < b.getDate())) age--;
    return age;
  }
});
