async function fetchWithRefresh(input, init = {}) {
  const res = await fetch(input, { credentials: 'include', ...init });
  if (res.status !== 401) return res;
  const r = await fetch('/auth/refresh', { method: 'POST', credentials: 'include' });
  if (!r.ok) return res;
  return fetch(input, { credentials: 'include', ...init });
}

document.addEventListener('DOMContentLoaded', async () => {
  await loadBlockedUsers();
});

function escapeHtml(s = '') {
  return String(s).replace(/[&<>"']/g, m => ({
    '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;'
  }[m]));
}

async function loadBlockedUsers() {
  const list = document.getElementById('blockedUsersList');
  try {
    const res = await fetchWithRefresh('/friendship/blocks');
    if (!res.ok) throw new Error(await res.text() || 'Falha ao carregar');
    const blocked = await res.json();

    if (!Array.isArray(blocked) || !blocked.length) {
      list.innerHTML = `<div class="empty">Você não bloqueou nenhum usuário ainda.</div>`;
      return;
    }

    list.innerHTML = blocked.map(u => `
      <article class="blocked-item" data-user-id="${u.id}">
        <div class="blocked-left">
          <img class="blocked-avatar" src="${u.profile_picture || '/img/user_placeholder.png'}" alt="${escapeHtml(u.username || 'Usuário')}">
          <span class="blocked-name">${escapeHtml(u.username || 'Usuário')}</span>
        </div>
        <button class="btn btn-danger" onclick="unblockUser(${Number(u.id)})">
          <i class="fa-solid fa-ban"></i> Desbloquear
        </button>
      </article>
    `).join('');
  } catch (e) {
    list.innerHTML = `<div class="empty empty--error">Erro: ${escapeHtml(e.message || 'Não foi possível carregar')}</div>`;
  }
}

async function unblockUser(blockedUserId) {
  if (!confirm('Deseja desbloquear este usuário?')) return;
  const list = document.getElementById('blockedUsersList');
  list.insertAdjacentHTML('afterbegin', `
    <div class="skeleton" id="unblockSkeleton">
      <div class="skeleton__line"></div>
      <div class="skeleton__line"></div>
    </div>
  `);
  try {
    const res = await fetchWithRefresh(`/friendship/blocks/${blockedUserId}`, { method: 'DELETE' });
    if (!res.ok) throw new Error(await res.text() || 'Falha ao desbloquear');
    await loadBlockedUsers();
  } catch (e) {
    alert(e.message);
  } finally {
    document.getElementById('unblockSkeleton')?.remove();
  }
}