// middlewares/validationMiddleware.js
const { body, param, query, validationResult } = require('express-validator');

/**
 * Política de senha simples e clara:
 * - mínimo 8 caracteres
 * - pelo menos 1 número
 * (ajuste aqui se quiser exigir maiúscula/símbolo)
 */
const PASSWORD_POLICY = {
  minLength: 8,
  minNumbers: 1
};

const validate = (method) => {
  switch (method) {
    // ==== AUTH ====
    case 'register':
      return [
        body('username')
          .trim()
          .notEmpty().withMessage('Informe um nome de usuário.')
          .isLength({ min: 3, max: 50 }).withMessage('Usuário deve ter entre 3 e 50 caracteres.'),
        body('email')
          .trim()
          .notEmpty().withMessage('Informe um e-mail.')
          .isEmail().withMessage('E-mail inválido.'),
        body('password')
          .notEmpty().withMessage('Informe uma senha.')
          .isLength({ min: PASSWORD_POLICY.minLength }).withMessage(`A senha deve ter pelo menos ${PASSWORD_POLICY.minLength} caracteres.`)
          .matches(/\d/).withMessage('A senha deve conter pelo menos 1 número.'),
        body('gender')
          .optional({ nullable: true })
          .isIn(['male', 'female', 'other']).withMessage('Gênero inválido.')
      ];

    case 'login':
      return [
        body('email')
          .trim()
          .notEmpty().withMessage('Informe um e-mail.')
          .isEmail().withMessage('E-mail inválido.'),
        body('password')
          .notEmpty().withMessage('Informe a senha.')
      ];

    // ==== AMIZADE/BLOQUEIO (seus casos existentes) ====
    case 'sendRequest':
      return [
        body('friendId')
          .notEmpty().withMessage('friendId é obrigatório.')
          .custom(v => {
            if (typeof v === 'number' && Number.isInteger(v) && v > 0) return true;
            if (typeof v === 'string') {
              if (/^\d$/.test(v)) return true;
              if (/^user_\d$/.test(v)) return true;
            }
            throw new Error('friendId inválido. Use número ou "user_123".');
          }),
        body('receiverUsername').optional().isString().trim().escape()
      ];

    case 'respondRequest':
      return [
        param('requestId').isInt().withMessage('ID da solicitação inválido.'),
        body('action')
          .isIn(['accepted', 'rejected'])
          .withMessage('A ação deve ser "accepted" ou "rejected".')
      ];

    case 'blockUser':
      return [
        body('targetId').isInt().withMessage('ID do usuário alvo inválido.')
      ];

    // ==== exemplo extra (caso use em /chat) ====
    case 'chatEnter':
      return [
        query('roomId')
          .matches(/^[a-zA-Z0-9-_]{6,64}$/).withMessage('roomId inválido.'),
        query('sessionId')
          .matches(/^[a-zA-Z0-9-_]{6,64}$/).withMessage('sessionId inválido.')
      ];

    default:
      return [];
  }
};

/** Formatador de erros amigável para o front */
function formatValidationErrors(result) {
  const list = result.array({ onlyFirstError: true }).map(e => {
    let hint;
    if (e.path === 'password') {
      hint = `Use ao menos ${PASSWORD_POLICY.minLength} caracteres e inclua números (ex.: "minhaSenha123").`;
    }
    if (e.path === 'email') {
      hint = 'Verifique se há "@" e domínio válido (ex.: nome@site.com).';
    }
    if (e.path === 'username') {
      hint = 'Use de 3 a 50 caracteres (letras, números, _.-).';
    }
    return {
      field: e.path,
      message: e.msg,
      value: e.value,
      location: e.location,
      hint
    };
  });

  const fields = list.reduce((acc, cur) => {
    acc[cur.field] = cur.message;
    return acc;
  }, {});

  const summary = list.map(e => `${e.field}: ${e.message}`).join(' | ');

  return { summary, errors: list, fields };
}

const handleValidation = (req, res, next) => {
  const result = validationResult(req);
  if (result.isEmpty()) return next();

  const payload = formatValidationErrors(result);

  // 422 = Unprocessable Entity (melhor que 400 pra erro de validação)
  return res.status(422).json({
    ok: false,
    code: 'VALIDATION_ERROR',
    ...payload
  });
};

module.exports = {
  validate,
  handleValidation
};
