const multer = require('multer');
const path = require('path');
const fs = require('fs');
const crypto = require('crypto');

function getUserIdFromReq(req) {
  if (req.user && req.user.id) return Number(req.user.id);
  throw new Error('Usuário não autenticado');
}

const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    let userId;
    try {
      userId = getUserIdFromReq(req);
    } catch (e) {
      return cb(e);
    }

    const dir = path.join(__dirname, '../public/uploads', String(userId), 'profile');
    fs.mkdir(dir, { recursive: true }, (err) => {
      if (err) return cb(err);
      cb(null, dir);
    });
  },

  filename: function (req, file, cb) {
    const ext = (path.extname(file.originalname) || '').toLowerCase();
    const name = 'avatar-' + crypto.randomBytes(8).toString('hex') + '-' + Date.now() + ext;
    cb(null, name);
  }
});

const allowedTypes = new Set([
  'image/jpeg',
  'image/png',
  'image/webp'
]);

function fileFilter(req, file, cb) {
  if (allowedTypes.has(file.mimetype)) {
    return cb(null, true);
  }
  cb(new Error('Apenas imagens JPEG, PNG ou WEBP são permitidas.'), false);
}

const upload = multer({
  storage,
  fileFilter,
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB
    files: 1
  }
});

module.exports = upload;
