// middlewares/galleryUpload.js
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const crypto = require('crypto');

function getUserIdFromReq(req) {
  if (req.user && req.user.id) return Number(req.user.id);
  throw new Error('Usuário não autenticado');
}

const MIME_EXT = {
  'image/jpeg': '.jpg',
  'image/png':  '.png',
  'image/webp': '.webp',
  'image/gif':  '.gif',
};

const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    let userId;
    try {
      userId = getUserIdFromReq(req);
    } catch (e) {
      return cb(e);
    }
    const uploadPath = path.join(__dirname, '../public/uploads', String(userId), 'gallery');
    fs.mkdir(uploadPath, { recursive: true }, (err) => {
      if (err) return cb(err);
      cb(null, uploadPath);
    });
  },
  filename: function (req, file, cb) {
    const extFromMime = MIME_EXT[file.mimetype] || (path.extname(file.originalname) || '').toLowerCase();
    const safeName = crypto.randomBytes(16).toString('hex') + '-' + Date.now() + extFromMime;
    cb(null, safeName);
  }
});

const allowedTypes = new Set(Object.keys(MIME_EXT));

function fileFilter(req, file, cb) {
  if (allowedTypes.has(file.mimetype)) return cb(null, true);
  cb(new Error('Tipo de arquivo não suportado. Use JPEG, PNG, WEBP ou GIF.'), false);
}

const upload = multer({
  storage,
  fileFilter,
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB por arquivo
    // ⚠️ Não defina "files" aqui para não travar o .array(...).
    // Se quiser, pode pôr algo como files: 12.
  }
});

module.exports = upload;
