const db = require('../models/db');
const path = require('path');

let io;

function initialize(socketIo) {
  io = socketIo;

  io.on('connection', (socket) => {
    // req.user fica no handshake (middleware do socket valida token)
    // este "socket.user" deve ser populado pelo seu middleware do socket
    try {
      if (!socket.user || !socket.user.id) {
        console.warn('Socket sem user injetado — verifique middleware do socket.');
        return;
      }
      socket.join(`user_${socket.user.id}`);
    } catch (e) {
      console.warn('Falha ao associar sala do usuário:', e);
    }

    socket.on('disconnect', () => { });
  });
}

exports.initialize = initialize;

exports.showRoom = (req, res) => {
  res.sendFile(path.join(__dirname, '../views/chatFriend.html'));
};

exports.getFriendData = async (req, res) => {
  try {
    const friendId = Number(req.params.friendId);
    const rows = await db.query('SELECT * FROM users WHERE id = ?', [friendId]);
    const list = Array.isArray(rows) && Array.isArray(rows[0]) ? rows[0] : rows;
    const friend = Array.isArray(list) ? list[0] : list;
    if (!friend) return res.status(404).json({ error: 'Amigo não encontrado' });
    res.json(friend);
  } catch (error) {
    console.error('Erro ao buscar dados do amigo:', error);
    res.status(500).json({ error: 'Erro interno do servidor' });
  }
};

exports.getFriendshipMeta = async (req, res) => {
  try {
    const currentUserId = req.user.id;
    const friendId = Number(req.params.friendId);
    const rows = await db.query(
      `SELECT created_at
         FROM friendships
        WHERE (user1_id=? AND user2_id=?) OR (user1_id=? AND user2_id=?)
        LIMIT 1`,
      [currentUserId, friendId, friendId, currentUserId]
    );
    const list = Array.isArray(rows) && Array.isArray(rows[0]) ? rows[0] : rows;
    const rec = Array.isArray(list) ? list[0] : list;
    res.json({ friendship_date: rec?.created_at || null });
  } catch (e) {
    console.error('getFriendshipMeta error:', e);
    res.status(500).json({ error: 'Erro ao buscar metadata da amizade' });
  }
};

exports.sendMessage = async (req, res) => {
  const senderId = req.user.id;
  const { receiverId, message } = req.body;

  if (!receiverId || !message) {
    return res.status(400).json({ error: 'Parâmetros ausentes' });
  }

  try {
    // Inserir a mensagem
    const result = await db.query(
      `INSERT INTO friend_messages (sender_id, receiver_id, message, sent_at, is_read)
       VALUES (?, ?, ?, NOW(), 0)`,
      [senderId, receiverId, message]
    );

    // Emitir para ambas as salas dos usuários
    const payload = {
      id: (result.insertId || (Array.isArray(result) && result[0]?.insertId)),
      sender_id: senderId,
      receiver_id: Number(receiverId),
      message,
      // ajuda o front a exibir o horário imediatamente
      sent_at: new Date().toISOString(),
      is_read: 0
    };

    if (io) {
      io.to(`user_${senderId}`).emit('new_message', payload);
      io.to(`user_${receiverId}`).emit('new_message', payload);
    }

    res.status(201).json({ success: true, id: payload.id });
  } catch (error) {
    console.error('Erro ao enviar mensagem:', error);
    res.status(500).json({ error: 'Erro ao enviar mensagem' });
  }
};

exports.getMessages = async (req, res) => {
  const currentUserId = req.user.id;
  const friendId = Number(req.params.friend_id);

  try {
    const result = await db.query(`
      SELECT id, sender_id, receiver_id, message, sent_at, is_read
      FROM friend_messages
      WHERE 
        (sender_id = ? AND receiver_id = ?)
        OR
        (sender_id = ? AND receiver_id = ?)
      ORDER BY sent_at ASC
    `, [currentUserId, friendId, friendId, currentUserId]);

    const rows = Array.isArray(result) && Array.isArray(result[0]) ? result[0] : (Array.isArray(result) ? result : [result]);
    res.json(rows);
  } catch (error) {
    console.error('Erro ao buscar mensagens:', error);
    res.status(500).json({ error: 'Erro ao buscar mensagens' });
  }
};

exports.markAsRead = async (req, res) => {
  try {
    const currentUserId = req.user.id;
    const friendId = Number(req.params.friendId);
    if (!friendId) return res.status(400).json({ error: 'friendId inválido' });

    await db.query(
      `UPDATE friend_messages
         SET is_read = 1
       WHERE sender_id = ? AND receiver_id = ? AND is_read = 0`,
      [friendId, currentUserId]
    );

    // Notifica o amigo/remetente que as mensagens dele foram lidas por "currentUserId"
    if (io) {
      io.to(`user_${friendId}`).emit('messages_read', {
        by: currentUserId,
        at: new Date().toISOString()
      });
    }

    res.status(200).json({ message: 'Mensagens marcadas como lidas' });
  } catch (e) {
    console.error('markAsRead error:', e);
    res.status(500).json({ error: e.message || 'Erro ao marcar mensagens' });
  }
};

// NOVO: contador de não lidas vindas do amigo
exports.getUnreadCount = async (req, res) => {
  try {
    const currentUserId = req.user.id;
    const friendId = Number(req.params.friendId);
    if (!friendId) return res.status(400).json({ error: 'friendId inválido' });

    const result = await db.query(
      `SELECT COUNT(*) AS unread
       FROM friend_messages
       WHERE sender_id = ? AND receiver_id = ? AND is_read = 0`,
      [friendId, currentUserId]
    );

    const rows = Array.isArray(result) && Array.isArray(result[0]) ? result[0] : result;
    const unread = Array.isArray(rows) && rows[0]?.unread != null ? Number(rows[0].unread) : Number(rows.unread || 0);

    res.status(200).json({ unread });
  } catch (e) {
    console.error('getUnreadCount error:', e);
    res.status(500).json({ error: e.message || 'Erro ao buscar não lidas' });
  }
};

exports.getTotalUnread = async (req, res) => {
  try {
    const currentUserId = req.user.id;
    const result = await db.query(
      `SELECT COUNT(*) AS total
         FROM friend_messages
        WHERE receiver_id = ? AND is_read = 0`,
      [currentUserId]
    );

    const rows = Array.isArray(result) && Array.isArray(result[0]) ? result[0] : result;
    const total = Array.isArray(rows) ? Number(rows[0]?.total || 0) : Number(rows.total || 0);

    res.json({ total });
  } catch (e) {
    console.error('getTotalUnread error:', e);
    res.status(500).json({ error: 'Erro ao calcular total de não lidas' });
  }
};
