// C:\src\Clientes\Datetime\app.js
require('dotenv').config();
const express = require('express');
const path = require('path');
const fs = require('fs');

const helmet = require('helmet');
const cors = require('cors');
const compression = require('compression');
const cookieParser = require('cookie-parser');
const rateLimit = require('express-rate-limit');
const hpp = require('hpp');
const morgan = require('morgan');
const multer = require('multer');

const errorHandler = require('./middlewares/errorHandler');
const { requireAuth } = require('./middlewares/authMiddleware');

const lobbyRoutes = require('./routes/lobbyRoutes');
const authRoutes = require('./routes/authRoutes');
const dashboardRoutes = require('./routes/dashboardRoutes');
const blockRoutes = require('./routes/blockRoutes');
const friendshipRoutes = require('./routes/friendshipRoutes');
const chatFriendsRoutes = require('./routes/chatFriendsRoutes');
const chatHistoryRoutes = require('./routes/chatHistoryRoutes');
const explorerRoute = require('./routes/explorerRoute');
const geoRoutes = require('./routes/geoRoutes');

const app = express();
const isProd = process.env.NODE_ENV === 'production';
const ORIGINS = (process.env.CORS_ORIGINS || 'http://localhost:3001')
  .split(',').map(s => s.trim());

// Infra/Sec
app.disable('x-powered-by');
app.set('trust proxy', 1);
app.use(morgan(isProd ? 'combined' : 'dev'));
app.use(cookieParser());
app.use(helmet({
  crossOriginResourcePolicy: { policy: 'cross-origin' },
  contentSecurityPolicy: {
    useDefaults: true,
    directives: {
      "default-src": ["'self'"],
      "connect-src": ["'self'", "https://servicodados.ibge.gov.br"],
      "img-src": ["'self'", "data:", "blob:"],
      "style-src": ["'self'", "'unsafe-inline'", "https://cdnjs.cloudflare.com"],
      "script-src": ["'self'", "'unsafe-inline'"],
      "font-src": ["'self'", "https://cdnjs.cloudflare.com"]
    }
  }
}));
app.use(hpp());
app.use(cors({
  origin: (origin, cb) => {
    if (!origin) return cb(null, true);
    return ORIGINS.includes(origin) ? cb(null, true) : cb(new Error('Not allowed by CORS'));
  },
  credentials: true,
  methods: ['GET','POST','PUT','PATCH','DELETE','OPTIONS'],
  allowedHeaders: ['Origin','X-Requested-With','Content-Type','Accept','Authorization','X-CSRF-Token']
}));
app.options('*', cors());

app.use(express.json({ limit: '200kb' }));
app.use(express.urlencoded({ extended: true, limit: '200kb' }));
app.use(compression());

// Static (com cache)
const staticOpts = { maxAge: isProd ? '1d' : 0, immutable: isProd };
app.use(express.static(path.join(__dirname, 'public'), staticOpts));
app.use('/static', express.static(path.join(__dirname, 'public'), staticOpts));
app.use('/uploads', express.static(path.join(__dirname, 'public/uploads'), { dotfiles: 'deny' }));

// Helper
function sendFileWithErrorHandling(res, filePath) {
  fs.access(filePath, fs.constants.F_OK, (err) => {
    if (err) return res.status(404).send(`Arquivo não encontrado: ${path.basename(filePath)}`);
    res.sendFile(filePath);
  });
}

// Rotas públicas (aplique rate-limit forte em auth)
const authLimiter = rateLimit({
  windowMs: 15 * 60 * 1000,
  max: 100,
  standardHeaders: true,
  legacyHeaders: false
});
app.use('/auth', authLimiter, authRoutes);

// ========= VIEWS autenticadas (coloque ANTES de montar o router /friendship) =========

// Amigos — VIEW canônica
app.get('/friends', requireAuth, (req, res) =>
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'friends.html'))
);

// Amigos — compatibilidade por content negotiation em /friendship/friends
// Se a navegação for de browser (aceita text/html), serve a página; senão, delega pro router (JSON).
app.get('/friendship/friends', requireAuth, (req, res, next) => {
  const accept = req.headers.accept || '';
  const secFetchMode = req.headers['sec-fetch-mode'] || '';
  const isNavigate = secFetchMode === 'navigate' || accept.includes('text/html');
  if (isNavigate) {
    return sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'friends.html'));
  }
  return next();
});

// Página de solicitações de amizade
app.get('/friends/requests', requireAuth, (req, res) => {
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'friendRequests.html'));
});

// Alias legado — mantém query string e método
app.get('/friends/list', requireAuth, (req, res) => {
  const qs = req.url.includes('?') ? req.url.slice(req.url.indexOf('?')) : '';
  res.redirect(307, `/friendship/friends${qs}`);
});

// Outras VIEWS já existentes
app.get('/amigos/ver/:id', requireAuth, (req, res) =>
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'friendProfile.html'))
);
app.get('/chat/historico/:id', requireAuth, (req, res) =>
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'chatHistory.html'))
);
app.get('/explorer/ver/:id', requireAuth, (req, res) =>
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'explorerProfile.html'))
);
app.get('/results', requireAuth, (req, res) =>
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'results.html'))
);
app.get('/lobby', requireAuth, (req, res) =>
  sendFileWithErrorHandling(res, path.join(__dirname, 'views', 'lobby.html'))
);

app.get('/chat', requireAuth, (req, res) => {
  const { roomId, sessionId } = req.query;
  const rx = /^[a-zA-Z0-9-_]{6,64}$/;
  if (!rx.test(roomId || '') || !rx.test(sessionId || '')) {
    return res.redirect('/?error=missing_or_invalid_chat_params');
  }
  const chatPath = path.join(__dirname, 'views', 'chat.html');
  fs.access(chatPath, fs.constants.F_OK, (err) => {
    if (err) return res.status(404).send('Página de chat não encontrada');
    res.sendFile(chatPath);
  });
});

// ========= Routers autenticados =========
// IMPORTANTE: montar /friendship DEPOIS do content-negotiation acima
app.use('/lobby', requireAuth, lobbyRoutes);
app.use('/block', requireAuth, blockRoutes);
app.use('/friendship', requireAuth, friendshipRoutes);
app.use('/chatFriends', requireAuth, chatFriendsRoutes);
app.use('/chatHistory', requireAuth, chatHistoryRoutes);
app.use('/dashboard', requireAuth, dashboardRoutes);
app.use('/explorer', requireAuth, explorerRoute);
app.use('/geo', geoRoutes);

// Raiz
app.get('/', (req, res) => {
  if (req.cookies?.token) return res.redirect('/dashboard/rooms');
  res.redirect('/auth/login');
});

// 404 e error handler (inclui tratamento multer)
app.use((req, res) => res.status(404).json({ error: 'Not found' }));
app.use((err, req, res, next) => {
  if (res.headersSent) return next(err);
  if (err instanceof multer.MulterError) {
    const map = {
      'LIMIT_FILE_SIZE': 'Arquivo excede o limite permitido.',
      'LIMIT_FILE_COUNT': 'Muitos arquivos enviados.',
      'LIMIT_UNEXPECTED_FILE': 'Campo de arquivo inesperado.'
    };
    const msg = map[err.code] || 'Erro no upload do arquivo.';
    return res.status(400).json({ error: msg, code: err.code });
  }
  if (/não autenticad|não autorizad/i.test(err.message)) {
    return res.status(401).json({ error: 'Não autenticado' });
  }
  if (/Tipo de arquivo não suportado/i.test(err.message)) {
    return res.status(415).json({ error: err.message });
  }
  return next(err);
});
app.use(errorHandler);

module.exports = app;
